<?php
declare(strict_types=1);

namespace App\Config;

use DateTimeImmutable;
use PDO;
use PDOException;
use PDOStatement;
use RuntimeException;
use Throwable;

require_once __DIR__ . '/app.php';

/**
 * Database access layer powered by PDO with logging and error handling.
 */
final class Database
{
    private const DB_HOST = 'localhost';
    private const DB_PORT = 3306;
    private const DB_NAME = 'helpdesk_sinarco';
    private const DB_USER = 'root';
    private const DB_PASS = 'root';
    private const DB_COLLATION = 'utf8mb4_unicode_ci';
    private const DB_CHARSET = 'utf8mb4';

    private static ?PDO $connection = null;

    private function __construct()
    {
    }

    public static function connection(): PDO
    {
        if (self::$connection instanceof PDO) {
            return self::$connection;
        }

        $dsn = sprintf(
            'mysql:host=%s;port=%d;dbname=%s;charset=%s',
            self::DB_HOST,
            self::DB_PORT,
            self::DB_NAME,
            self::DB_CHARSET
        );

        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_PERSISTENT => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => sprintf(
                'SET NAMES %s COLLATE %s',
                self::DB_CHARSET,
                self::DB_COLLATION
            ),
        ];

        try {
            self::$connection = new PDO($dsn, self::DB_USER, self::DB_PASS, $options);
        } catch (PDOException $exception) {
            self::log('ERROR', 'Connection failed: ' . $exception->getMessage());
            throw new RuntimeException('Unable to connect to the database.', 0, $exception);
        }

        return self::$connection;
    }

    public static function prepare(string $sql): PDOStatement
    {
        try {
            return self::connection()->prepare($sql);
        } catch (PDOException $exception) {
            self::log('ERROR', 'Prepare failed: ' . $exception->getMessage() . ' | SQL: ' . $sql);
            throw new RuntimeException('Unable to prepare the statement.', 0, $exception);
        }
    }

    public static function run(string $sql, array $parameters = []): PDOStatement
    {
        $statement = self::prepare($sql);

        try {
            $statement->execute($parameters);
        } catch (PDOException $exception) {
            self::log(
                'ERROR',
                'Execute failed: ' . $exception->getMessage() . ' | SQL: ' . $sql . ' | Params: ' . json_encode($parameters)
            );
            throw new RuntimeException('Unable to execute the statement.', 0, $exception);
        }

        return $statement;
    }

    public static function testConnection(): bool
    {
        try {
            self::connection()->query('SELECT 1');

            if (AppConfig::isDebug()) {
                self::log('INFO', 'Database connection test succeeded.');
            }

            return true;
        } catch (Throwable $exception) {
            self::log('ERROR', 'Database connection test failed: ' . $exception->getMessage());
            return false;
        }
    }

    private static function log(string $level, string $message): void
    {
        $timestamp = (new DateTimeImmutable())->format('Y-m-d H:i:s');
        $logDirectory = dirname(__DIR__, 2) . '/storage/logs';
        $logFile = $logDirectory . '/database.log';

        if (!is_dir($logDirectory)) {
            mkdir($logDirectory, 0775, true);
        }

        $line = sprintf('[%s] %s %s%s', $timestamp, $level, $message, PHP_EOL);
        file_put_contents($logFile, $line, FILE_APPEND);
    }
}

