<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Repositories\AdminAuditRepository;
use App\Repositories\CategoryRepository;
use App\Repositories\QueueRepository;
use App\Repositories\RoleRepository;
use App\Repositories\SlaPolicyRepository;
use App\Repositories\ThemeRepository;
use App\Repositories\LookupRepository;
use App\Services\AdminService;
use App\Services\AuthService;
use App\Services\CategoryService;
use App\Services\FieldService;
use App\Services\QueueService;
use RuntimeException;

final class AdminController extends Controller
{
    public function __construct(
        private readonly AuthService $authService,
        private readonly AdminService $adminService,
        private readonly CategoryRepository $categories,
        private readonly CategoryService $categoryService,
        private readonly SlaPolicyRepository $slaPolicies,
        private readonly ThemeRepository $themes,
        private readonly QueueRepository $queues,
        private readonly RoleRepository $roles,
        private readonly LookupRepository $lookups,
        private readonly AdminAuditRepository $audits,
        private readonly FieldService $fieldService,
        private readonly QueueService $queueService
    ) {
    }

    public function index(): string
    {
        header('Location: ' . app_url('admin/categories'));
        exit;
    }

    public function sla(): string
    {
        $user = $this->requireGestor();
        $data = $this->baseData($user, 'sla');

        return $this->render('admin/slas', array_merge($data, [
            'slaPolicies' => $this->slaPolicies->all(),
            'priorities' => $this->lookups->priorities(),
            'categories' => $this->categories->all(),
        ]));
    }

    public function theme(): string
    {
        $user = $this->requireGestor();
        $data = $this->baseData($user, 'theme');
        $theme = $this->themes->findByContext('global') ?? $this->themes->findDefault() ?? [
            'primary_color' => '#ff7a18',
            'secondary_color' => '#6c63ff',
            'dark_color' => '#1f1f1f',
            'accent_color' => '#37474f',
            'logo_text' => 'HelpDesk',
        ];

        return $this->render('admin/theme', array_merge($data, [
            'theme' => $theme,
        ]));
    }

    public function uploadThemeLogo(): void
    {
        $user = $this->requireGestor();
        csrf_validate();

        try {
            $this->adminService->updateThemeLogo($_FILES['logo'] ?? [], $user);
            $this->flash('admin_flash', ['success' => 'Logo atualizada com sucesso.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        header('Location: ' . app_url('admin/theme'));
        exit;
    }

    public function removeThemeLogo(): void
    {
        $user = $this->requireGestor();
        csrf_validate();

        try {
            $this->adminService->removeThemeLogo($user);
            $this->flash('admin_flash', ['success' => 'Logo removida com sucesso.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        header('Location: ' . app_url('admin/theme'));
        exit;
    }

    public function users(): string
    {
        $user = $this->requireGestor();
        $data = $this->baseData($user, 'users');

        return $this->render('admin/users', array_merge($data, [
            'users' => $this->lookups->activeUsers(),
            'roles' => $this->roles->all(),
            'queues' => $this->queues->all(),
        ]));
    }

    public function handle(): void
    {
        $user = $this->requireGestor();
        $action = (string) ($_POST['action'] ?? '');

        try {
            csrf_validate();
            match ($action) {
                'create_category' => $this->categoryService->create($_POST, $user),
                'create_queue' => $this->queueService->create($_POST, $user),
                'create_field' => $this->fieldService->create($_POST, $user),
                'create_sla' => $this->adminService->createSlaPolicy($_POST, $user),
                'update_theme' => $this->adminService->updateTheme($_POST, $user),
                default => throw new RuntimeException('Ação administrativa desconhecida.'),
            };

            $this->flash('admin_flash', ['success' => 'Alteração registrada com sucesso.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        header('Location: ' . app_url($this->redirectPath($action)));
        exit;
    }

    private function requireGestor(): array
    {
        $user = $this->authService->user();
        if ($user === null || ($user['role_slug'] ?? '') !== 'gestor') {
            throw new RuntimeException('Acesso restrito aos gestores.');
        }

        return $user;
    }

    private function flash(string $key, array $value): void
    {
        $_SESSION[$key] = $value;
    }

    /**
     * @return array<string, mixed>
     */
    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }

    /**
     * @param array<string, mixed> $user
     * @return array<string, mixed>
     */
    private function baseData(array $user, string $section): array
    {
        return [
            'flash' => $this->pullFlash('admin_flash', []),
            'errors' => $this->pullFlash('admin_errors', []),
            'pageTitle' => 'Administração',
            'authUser' => $user,
            'adminSection' => $section,
        ];
    }

    private function redirectPath(string $action): string
    {
        return match ($action) {
            'create_category' => 'admin/categories',
            'create_queue' => 'admin/queues',
            'create_field' => 'admin/fields',
            'create_sla' => 'admin/slas',
            'update_theme' => 'admin/theme',
            default => 'admin/categories',
        };
    }
}
