<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class ReportRepository
{
    /**
     * @return array<int, array<string, mixed>>
     */
    public function categoryPriorityMatrix(?string $fromDate, ?string $toDate): array
    {
        $params = [];
        $where = $this->dateClause('t.created_at', $fromDate, $toDate, $params);

        $sql = <<<SQL
            SELECT
                COALESCE(cat.name, 'Sem subcategoria') AS category_name,
                p.name AS priority_name,
                COUNT(*) AS total
            FROM tickets t
            LEFT JOIN categories cat ON cat.id = t.category_id
            INNER JOIN priorities p ON p.id = t.priority_id
            WHERE 1 = 1 {$where}
            GROUP BY category_name, priority_name, p.sort_order
            ORDER BY category_name ASC, p.sort_order ASC, priority_name ASC
        SQL;

        $statement = Database::run($sql, $params);

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function typeSubcategoryBreakdown(?string $fromDate, ?string $toDate): array
    {
        $params = [];
        $where = $this->dateClause('t.created_at', $fromDate, $toDate, $params);

        $sql = <<<SQL
            SELECT
                UPPER(t.main_category) AS main_category,
                COALESCE(cat.name, 'Sem subcategoria') AS subcategory,
                COUNT(*) AS total
            FROM tickets t
            LEFT JOIN categories cat ON cat.id = t.category_id
            WHERE 1 = 1 {$where}
            GROUP BY t.main_category, subcategory
            ORDER BY t.main_category ASC, total DESC
        SQL;

        $statement = Database::run($sql, $params);

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array<string, ?float>
     */
    public function averageTimes(?string $fromDate, ?string $toDate): array
    {
        $params = [];
        $where = $this->dateClause('t.created_at', $fromDate, $toDate, $params);

        $sql = <<<SQL
            SELECT
                AVG(TIMESTAMPDIFF(MINUTE, t.created_at, resp.first_response_at)) AS avg_response,
                AVG(CASE WHEN t.resolved_at IS NOT NULL THEN TIMESTAMPDIFF(MINUTE, t.created_at, t.resolved_at) END) AS avg_resolution
            FROM tickets t
            LEFT JOIN (
                SELECT
                    sh.ticket_id,
                    MIN(sh.changed_at) AS first_response_at
                FROM ticket_status_history sh
                INNER JOIN statuses s ON s.id = sh.to_status_id
                WHERE s.slug IN ('em-triagem', 'em-atendimento')
                GROUP BY sh.ticket_id
            ) resp ON resp.ticket_id = t.id
            WHERE 1 = 1 {$where}
        SQL;

        $statement = Database::run($sql, $params);
        $row = $statement->fetch(PDO::FETCH_ASSOC) ?: [];

        return [
            'avg_response' => $row['avg_response'] !== null ? (float) $row['avg_response'] : null,
            'avg_resolution' => $row['avg_resolution'] !== null ? (float) $row['avg_resolution'] : null,
        ];
    }

    /**
     * @return array<string, int>
     */
    public function slaCompliance(?string $fromDate, ?string $toDate): array
    {
        $params = [];
        $where = $this->dateClause('t.created_at', $fromDate, $toDate, $params);

        $sql = <<<SQL
            SELECT
                SUM(
                    CASE
                        WHEN t.sla_due_at IS NOT NULL
                             AND t.resolved_at IS NOT NULL
                             AND t.resolved_at <= t.sla_due_at
                        THEN 1 ELSE 0
                    END
                ) AS met,
                SUM(
                    CASE
                        WHEN t.sla_due_at IS NOT NULL
                             AND (
                                 (t.resolved_at IS NOT NULL AND t.resolved_at > t.sla_due_at)
                                 OR (t.resolved_at IS NULL AND NOW() > t.sla_due_at)
                             )
                        THEN 1 ELSE 0
                    END
                ) AS breached
            FROM tickets t
            WHERE 1 = 1 {$where}
        SQL;

        $statement = Database::run($sql, $params);
        $row = $statement->fetch(PDO::FETCH_ASSOC) ?: [];

        return [
            'met' => (int) ($row['met'] ?? 0),
            'breached' => (int) ($row['breached'] ?? 0),
        ];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function rankingByAssignee(?string $fromDate, ?string $toDate, int $limit = 5): array
    {
        $params = [];
        $where = $this->dateClause('t.created_at', $fromDate, $toDate, $params);

        $sql = <<<SQL
            SELECT
                COALESCE(u.name, 'Sem responsável') AS assignee_name,
                COUNT(*) AS total_tickets,
                AVG(CASE WHEN t.resolved_at IS NOT NULL THEN TIMESTAMPDIFF(MINUTE, t.created_at, t.resolved_at) END) AS avg_resolution
            FROM tickets t
            LEFT JOIN users u ON u.id = t.assignee_id
            WHERE t.assignee_id IS NOT NULL {$where}
            GROUP BY u.id, assignee_name
            ORDER BY total_tickets DESC
            LIMIT :limit
        SQL;

        $statement = Database::connection()->prepare($sql);
        foreach ($params as $name => $value) {
            $statement->bindValue(':' . $name, $value);
        }
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->execute();

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function rankingByCustomer(?string $fromDate, ?string $toDate, int $limit = 5): array
    {
        $params = [];
        $where = $this->dateClause('t.created_at', $fromDate, $toDate, $params);
        $customerExpression = "COALESCE(NULLIF(t.customer_name, ''), NULLIF(t.site_name, ''), 'Sem identificação')";

        $sql = <<<SQL
            SELECT
                {$customerExpression} AS customer_name,
                COUNT(*) AS total_tickets,
                AVG(
                    CASE
                        WHEN t.resolved_at IS NOT NULL
                        THEN TIMESTAMPDIFF(MINUTE, t.created_at, t.resolved_at)
                    END
                ) AS avg_resolution
            FROM tickets t
            WHERE 1 = 1 {$where}
            GROUP BY {$customerExpression}
            ORDER BY total_tickets DESC
            LIMIT :limit
        SQL;

        $statement = Database::connection()->prepare($sql);
        foreach ($params as $name => $value) {
            $statement->bindValue(':' . $name, $value);
        }
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->execute();

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @param array<string, mixed> $params
     */
    private function dateClause(string $column, ?string $fromDate, ?string $toDate, array &$params): string
    {
        $clauses = [];

        if ($fromDate !== null && $fromDate !== '') {
            $clauses[] = "{$column} >= :from_date";
            $params['from_date'] = $fromDate . ' 00:00:00';
        }

        if ($toDate !== null && $toDate !== '') {
            $clauses[] = "{$column} <= :to_date";
            $params['to_date'] = $toDate . ' 23:59:59';
        }

        if ($clauses === []) {
            return '';
        }

        return ' AND ' . implode(' AND ', $clauses);
    }
}
