<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class UserRepository
{
    public function findByEmail(string $email): ?array
    {
        $sql = <<<SQL
            SELECT u.*, r.slug AS role_slug, r.name AS role_name
            FROM users u
            INNER JOIN roles r ON r.id = u.role_id
            WHERE u.email = :email
              AND u.deleted_at IS NULL
            LIMIT 1
        SQL;

        $statement = Database::run($sql, ['email' => $email]);
        $user = $statement->fetch(PDO::FETCH_ASSOC);

        return $user === false ? null : $user;
    }

    public function findById(int $id): ?array
    {
        $sql = <<<SQL
            SELECT u.*, r.slug AS role_slug, r.name AS role_name
            FROM users u
            INNER JOIN roles r ON r.id = u.role_id
            WHERE u.id = :id
              AND u.deleted_at IS NULL
            LIMIT 1
        SQL;

        $statement = Database::run($sql, ['id' => $id]);
        $user = $statement->fetch(PDO::FETCH_ASSOC);

        return $user === false ? null : $user;
    }

    public function updateLastLogin(int $id): void
    {
        $sql = 'UPDATE users SET last_login_at = NOW() WHERE id = :id';
        Database::run($sql, ['id' => $id]);
    }

    /**
     * @param array<string, mixed> $data
     */
    public function create(array $data): int
    {
        $sql = <<<SQL
            INSERT INTO users (role_id, queue_id, name, email, password_hash, phone, is_active)
            VALUES (:role_id, :queue_id, :name, :email, :password_hash, :phone, :is_active)
        SQL;

        Database::run($sql, $data);
        return (int) Database::connection()->lastInsertId();
    }

    /**
     * @return array{rows: array<int, array<string, mixed>>, total: int}
     */
    public function paginate(?string $search, ?string $status, int $limit, int $offset): array
    {
        $conditions = ['u.deleted_at IS NULL'];
        $params = [];

        if ($search !== null && $search !== '') {
            $conditions[] = '(u.name LIKE :search OR u.email LIKE :search)';
            $params['search'] = '%' . $search . '%';
        }

        if ($status === 'active') {
            $conditions[] = 'u.is_active = 1';
        } elseif ($status === 'inactive') {
            $conditions[] = 'u.is_active = 0';
        }

        $where = 'WHERE ' . implode(' AND ', $conditions);

        $countSql = 'SELECT COUNT(*) FROM users u ' . $where;
        $countStatement = Database::prepare($countSql);
        foreach ($params as $key => $value) {
            $countStatement->bindValue(':' . $key, $value);
        }
        $countStatement->execute();
        $total = (int) $countStatement->fetchColumn();

        $sql = <<<SQL
            SELECT
                u.id,
                u.name,
                u.email,
                u.phone,
                u.role_id,
                u.queue_id,
                u.is_active,
                u.created_at,
                r.name AS role_name
            FROM users u
            INNER JOIN roles r ON r.id = u.role_id
            $where
            ORDER BY u.created_at DESC, u.id DESC
            LIMIT :limit OFFSET :offset
        SQL;

        $statement = Database::connection()->prepare($sql);
        foreach ($params as $key => $value) {
            $statement->bindValue(':' . $key, $value);
        }
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->bindValue(':offset', $offset, PDO::PARAM_INT);
        $statement->execute();

        return [
            'rows' => $statement->fetchAll(PDO::FETCH_ASSOC) ?: [],
            'total' => $total,
        ];
    }

    public function emailExists(string $email, ?int $ignoreId = null): bool
    {
        $sql = 'SELECT 1 FROM users WHERE email = :email AND deleted_at IS NULL';
        $params = ['email' => $email];

        if ($ignoreId !== null) {
            $sql .= ' AND id <> :ignore_id';
            $params['ignore_id'] = $ignoreId;
        }

        $statement = Database::run($sql, $params);
        return $statement->fetchColumn() !== false;
    }

    /**
     * @param array<string, mixed> $data
     */
    public function update(int $id, array $data): bool
    {
        if ($data === []) {
            return false;
        }

        $fields = [];
        foreach ($data as $column => $value) {
            $fields[] = sprintf('%s = :%s', $column, $column);
        }
        $fields[] = 'updated_at = NOW()';

        $sql = sprintf('UPDATE users SET %s WHERE id = :id AND deleted_at IS NULL', implode(', ', $fields));
        $data['id'] = $id;

        $statement = Database::run($sql, $data);
        return $statement->rowCount() > 0;
    }

    public function softDelete(int $id): bool
    {
        $sql = 'UPDATE users SET deleted_at = NOW(), updated_at = NOW() WHERE id = :id AND deleted_at IS NULL';
        $statement = Database::run($sql, ['id' => $id]);
        return $statement->rowCount() > 0;
    }

    public function updateStatus(int $id, int $isActive): bool
    {
        $sql = 'UPDATE users SET is_active = :is_active, updated_at = NOW() WHERE id = :id AND deleted_at IS NULL';
        $statement = Database::run($sql, [
            'id' => $id,
            'is_active' => $isActive,
        ]);

        return $statement->rowCount() > 0;
    }

    public function updatePassword(int $id, string $passwordHash): bool
    {
        $sql = 'UPDATE users SET password_hash = :password_hash, updated_at = NOW() WHERE id = :id AND deleted_at IS NULL';
        $statement = Database::run($sql, [
            'id' => $id,
            'password_hash' => $passwordHash,
        ]);

        return $statement->rowCount() > 0;
    }
}
