<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\AdminAuditRepository;
use App\Repositories\SlaPolicyRepository;
use App\Repositories\ThemeRepository;
use App\Repositories\UserRepository;
use RuntimeException;

final class AdminService
{
    public function __construct(
        private readonly SlaPolicyRepository $slaPolicies,
        private readonly ThemeRepository $themes,
        private readonly UserRepository $users,
        private readonly AdminAuditRepository $audit
    ) {
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed> $user
     */
    public function createSlaPolicy(array $input, array $user): void
    {
        $priorityId = $this->requiredInt($input['priority_id'] ?? null, 'Escolha uma prioridade.');
        $response = max(0, (int) ($input['response_minutes'] ?? 0));
        $resolution = max(0, (int) ($input['resolution_minutes'] ?? 0));
        if ($response === 0 && $resolution === 0) {
            throw new RuntimeException('Informe metas de resposta e/ou resolução.');
        }

        $clientType = trim((string) ($input['client_type'] ?? ''));
        if ($clientType !== '' && !in_array($clientType, ['interno', 'externo'], true)) {
            $clientType = '';
        }

        $policyId = $this->slaPolicies->create([
            'category_id' => $this->nullableInt($input['category_id'] ?? null),
            'priority_id' => $priorityId,
            'client_type' => $clientType !== '' ? $clientType : null,
            'response_minutes' => $response,
            'resolution_minutes' => $resolution,
            'sort_order' => (int) ($input['sort_order'] ?? 0),
            'is_active' => isset($input['is_active']) ? 1 : 0,
        ]);

        $this->log($user, 'sla_policy', 'create', $policyId, [
            'priority_id' => $priorityId,
            'client_type' => $clientType,
        ]);
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed> $user
     */
    public function updateTheme(array $input, array $user): void
    {
        $current = $this->themes->findByContext('global') ?? $this->themes->findDefault() ?? [];
        $data = [
            'primary_color' => $this->validateColor($input['primary_color'] ?? '#ff7a18'),
            'secondary_color' => $this->validateColor($input['secondary_color'] ?? '#6c63ff'),
            'dark_color' => $this->validateColor($input['dark_color'] ?? '#1f1f1f'),
            'accent_color' => $this->validateColor($input['accent_color'] ?? '#37474f'),
            'logo_text' => trim((string) ($input['logo_text'] ?? 'HelpDesk')),
            'logo_path' => $current['logo_path'] ?? null,
        ];

        $this->themes->save('global', $data);
        $this->log($user, 'theme', 'update', null, $data);
    }

    /**
     * Trata upload e otimização da logo do sistema.
     *
     * @param array<string, mixed> $file
     * @param array<string, mixed> $user
     */
    public function updateThemeLogo(array $file, array $user): void
    {
        if (!extension_loaded('gd')) {
            throw new RuntimeException('Biblioteca GD não disponível para processar imagens.');
        }

        $error = (int) ($file['error'] ?? UPLOAD_ERR_NO_FILE);
        if ($error !== UPLOAD_ERR_OK) {
            throw new RuntimeException('Falha no upload da logo (código ' . $error . ').');
        }

        $size = (int) ($file['size'] ?? 0);
        if ($size <= 0 || $size > 2 * 1024 * 1024) {
            throw new RuntimeException('Logo deve ter no máximo 2MB.');
        }

        $tmpName = (string) ($file['tmp_name'] ?? '');
        if ($tmpName === '' || !is_uploaded_file($tmpName)) {
            throw new RuntimeException('Upload inválido.');
        }

        $originalName = (string) ($file['name'] ?? '');
        $extension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
        if (!in_array($extension, ['jpg', 'jpeg', 'png'], true)) {
            throw new RuntimeException('Formatos permitidos: JPG, JPEG ou PNG.');
        }

        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = $finfo ? finfo_file($finfo, $tmpName) : null;
        if ($finfo) {
            finfo_close($finfo);
        }

        if (!in_array($mimeType, ['image/jpeg', 'image/png'], true)) {
            throw new RuntimeException('Arquivo enviado não é uma imagem válida.');
        }

        $imageInfo = @getimagesize($tmpName);
        if ($imageInfo === false) {
            throw new RuntimeException('Não foi possível ler a imagem enviada.');
        }
        [$width, $height] = $imageInfo;

        $source = $mimeType === 'image/png' ? @imagecreatefrompng($tmpName) : @imagecreatefromjpeg($tmpName);
        if (!$source) {
            throw new RuntimeException('Não foi possível processar a imagem enviada.');
        }

        [$targetWidth, $targetHeight] = $this->calculateLogoDimensions($width, $height, 220, 60);
        $destination = imagecreatetruecolor($targetWidth, $targetHeight);

        // Preserva transparência ao exportar em PNG.
        imagealphablending($destination, false);
        imagesavealpha($destination, true);
        $transparent = imagecolorallocatealpha($destination, 0, 0, 0, 127);
        imagefilledrectangle($destination, 0, 0, $targetWidth - 1, $targetHeight - 1, $transparent);

        if (!imagecopyresampled(
            $destination,
            $source,
            0,
            0,
            0,
            0,
            $targetWidth,
            $targetHeight,
            $width,
            $height
        )) {
            imagedestroy($source);
            imagedestroy($destination);
            throw new RuntimeException('Falha ao redimensionar a imagem.');
        }

        $filename = sprintf('logo_%d_%s.png', time(), bin2hex(random_bytes(4)));
        $relativePath = 'uploads/theme/' . $filename;
        $targetDirectory = BASE_PATH . '/public/uploads/theme';
        if (!is_dir($targetDirectory) && !mkdir($targetDirectory, 0775, true) && !is_dir($targetDirectory)) {
            imagedestroy($source);
            imagedestroy($destination);
            throw new RuntimeException('Não foi possível criar o diretório de logos.');
        }

        $fullPath = $targetDirectory . '/' . $filename;
        if (!imagepng($destination, $fullPath, 6)) {
            imagedestroy($source);
            imagedestroy($destination);
            throw new RuntimeException('Não foi possível salvar a logo processada.');
        }

        imagedestroy($source);
        imagedestroy($destination);

        $current = $this->themes->findByContext('global') ?? $this->themes->findDefault() ?? [];
        if (!empty($current['logo_path'])) {
            $this->deleteLogoFile((string) $current['logo_path']);
        }

        $this->themes->updateLogoPath('global', $relativePath);
        $this->log($user, 'theme', 'update_logo', null, ['logo_path' => $relativePath]);
    }

    public function removeThemeLogo(array $user): void
    {
        $current = $this->themes->findByContext('global') ?? $this->themes->findDefault();
        if ($current === null || empty($current['logo_path'])) {
            throw new RuntimeException('Nenhuma logo configurada para remover.');
        }

        $this->deleteLogoFile((string) $current['logo_path']);
        $this->themes->updateLogoPath('global', null);
        $this->log($user, 'theme', 'remove_logo', null);
    }

    /**
     * @return array{0:int,1:int}
     */
    private function calculateLogoDimensions(int $width, int $height, int $maxWidth, int $maxHeight): array
    {
        if ($width <= 0 || $height <= 0) {
            return [$maxWidth, $maxHeight];
        }

        $ratio = min(1.0, min($maxWidth / $width, $maxHeight / $height));
        $targetWidth = (int) max(1, round($width * $ratio));
        $targetHeight = (int) max(1, round($height * $ratio));

        return [$targetWidth, $targetHeight];
    }

    private function deleteLogoFile(string $relativePath): void
    {
        $cleanPath = ltrim($relativePath, '/');
        if ($cleanPath === '' || !str_starts_with($cleanPath, 'uploads/')) {
            return;
        }

        $fullPath = BASE_PATH . '/public/' . $cleanPath;
        if (is_file($fullPath)) {
            @unlink($fullPath);
        }
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed> $user
     */
    public function createUser(array $input, array $user): void
    {
        $name = trim((string) ($input['name'] ?? ''));
        $email = strtolower(trim((string) ($input['email'] ?? '')));
        $password = (string) ($input['password'] ?? '');
        $roleId = $this->requiredInt($input['role_id'] ?? null, 'Selecione o perfil do usuário.');

        if ($name === '' || $email === '') {
            throw new RuntimeException('Nome e e-mail são obrigatórios.');
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new RuntimeException('Informe um e-mail válido.');
        }

        if (strlen($password) < 6) {
            throw new RuntimeException('Senha deve ter pelo menos 6 caracteres.');
        }

        if ($this->users->findByEmail($email) !== null) {
            throw new RuntimeException('Já existe um usuário com este e-mail.');
        }

        $userId = $this->users->create([
            'role_id' => $roleId,
            'queue_id' => $this->nullableInt($input['queue_id'] ?? null),
            'name' => $name,
            'email' => $email,
            'password_hash' => password_hash($password, PASSWORD_BCRYPT),
            'phone' => trim((string) ($input['phone'] ?? '')) ?: null,
            'is_active' => isset($input['is_active']) ? 1 : 0,
        ]);

        $this->log($user, 'user', 'create', $userId, [
            'email' => $email,
            'role_id' => $roleId,
        ]);
    }

    private function nullableInt(mixed $value): ?int
    {
        if ($value === null || $value === '' || (int) $value === 0) {
            return null;
        }

        return (int) $value;
    }

    private function requiredInt(mixed $value, string $message): int
    {
        $intValue = $this->nullableInt($value);
        if ($intValue === null) {
            throw new RuntimeException($message);
        }

        return $intValue;
    }

    private function validateColor(string $color): string
    {
        $color = trim($color);
        if (!preg_match('/^#[0-9a-fA-F]{6}$/', $color)) {
            throw new RuntimeException('Informe cores no formato hexadecimal (#RRGGBB).');
        }

        return strtoupper($color);
    }

    /**
     * @param array<string, mixed> $user
     * @param array<string, mixed> $payload
     */
    private function log(array $user, string $entity, string $action, ?int $entityId, array $payload = []): void
    {
        $userId = isset($user['id']) ? (int) $user['id'] : 0;
        if ($userId === 0) {
            return;
        }

        \app_log('INFO', 'Admin action', [
            'user_id' => $userId,
            'entity' => $entity,
            'action' => $action,
            'entity_id' => $entityId,
        ]);

        $this->audit->log($userId, $entity, $action, $entityId, $payload);
    }
}
