<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Repositories\PriorityRepository;
use App\Services\AuthService;
use App\Services\SlaService;
use RuntimeException;

final class AdminSlasController extends Controller
{
    private const OLD_INPUT_KEY = 'admin_slas_old';

    public function __construct(
        private readonly AuthService $authService,
        private readonly SlaService $slaService,
        private readonly PriorityRepository $priorityRepository
    ) {
    }

    public function index(): string
    {
        $user = $this->requireGestor();
        $search = isset($_GET['q']) ? (string) $_GET['q'] : null;
        $status = isset($_GET['status']) ? (string) $_GET['status'] : null;
        $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;

        $result = $this->slaService->getPaginated($search, $status, $page);

        return $this->render('admin/slas/index', array_merge($this->baseData($user, 'sla'), [
            'slas' => $result['data'],
            'pagination' => $result['pagination'],
            'filters' => [
                'q' => $result['filters']['search'] ?? '',
                'status' => $result['filters']['status'] ?? 'all',
            ],
        ]));
    }

    public function create(): string
    {
        $user = $this->requireGestor();

        return $this->render('admin/slas/form', array_merge($this->baseData($user, 'sla'), [
            'formAction' => app_url('admin/slas/store'),
            'formTitle' => 'Novo SLA',
            'isEdit' => false,
            'slaRecord' => null,
            'priorities' => $this->priorityRepository->all(),
        ]));
    }

    public function store(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $input = $_POST ?? [];
        try {
            $this->slaService->create($input);
            $this->flash('admin_flash', ['success' => 'SLA criado com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/slas');
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/slas/create');
        }
    }

    public function edit(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['SLA inválido.']);
            $this->redirect('admin/slas');
        }

        $record = $this->slaService->find($id);
        if ($record === null) {
            $this->flash('admin_errors', ['SLA não encontrado.']);
            $this->redirect('admin/slas');
        }

        return $this->render('admin/slas/form', array_merge($this->baseData($user, 'sla'), [
            'formAction' => app_url('admin/slas/update?id=' . $id),
            'formTitle' => 'Editar SLA',
            'isEdit' => true,
            'slaRecord' => $record,
            'priorities' => $this->priorityRepository->all(),
        ]));
    }

    public function update(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['SLA inválido.']);
            $this->redirect('admin/slas');
        }

        $input = $_POST ?? [];
        try {
            $this->slaService->update($id, $input);
            $this->flash('admin_flash', ['success' => 'SLA atualizado com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/slas/edit?id=' . $id);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/slas/edit?id=' . $id);
        }
    }

    public function confirmDelete(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['SLA inválido.']);
            $this->redirect('admin/slas');
        }

        $record = $this->slaService->find($id);
        if ($record === null) {
            $this->flash('admin_errors', ['SLA não encontrado.']);
            $this->redirect('admin/slas');
        }

        return $this->render('admin/slas/delete', array_merge($this->baseData($user, 'sla'), [
            'slaRecord' => $record,
        ]));
    }

    public function delete(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['SLA inválido.']);
            $this->redirect('admin/slas');
        }

        try {
            $this->slaService->delete($id);
            $this->flash('admin_flash', ['success' => 'SLA removido com sucesso.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect('admin/slas');
    }

    public function toggleStatus(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        $activate = isset($_POST['status']) && (int) $_POST['status'] === 1;
        $redirect = isset($_POST['redirect']) ? (string) $_POST['redirect'] : 'admin/slas';

        if ($id <= 0) {
            $this->flash('admin_errors', ['SLA inválido.']);
            $this->redirect($redirect);
        }

        try {
            $this->slaService->toggleStatus($id, $activate);
            $this->flash('admin_flash', ['success' => 'Status atualizado.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    private function requireGestor(): array
    {
        $user = $this->authService->user();
        if ($user === null || ($user['role_slug'] ?? '') !== 'gestor') {
            throw new RuntimeException('Acesso restrito aos gestores.');
        }

        return $user;
    }

    private function flash(string $key, array $value): void
    {
        $_SESSION[$key] = $value;
    }

    /**
     * @param array<string, mixed> $user
     * @return array<string, mixed>
     */
    private function baseData(array $user, string $section): array
    {
        return [
            'flash' => $this->pullFlash('admin_flash', []),
            'errors' => $this->pullFlash('admin_errors', []),
            'old' => $this->pullOldInput(),
            'pageTitle' => 'Administração',
            'authUser' => $user,
            'adminSection' => $section,
        ];
    }

    /**
     * @return array<mixed>
     */
    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }

    private function rememberOldInput(array $input): void
    {
        $payload = $input;
        $payload['business_hours_only'] = isset($input['business_hours_only']) ? 1 : 0;
        $payload['escalation_enabled'] = isset($input['escalation_enabled']) ? 1 : 0;
        $payload['is_active'] = isset($input['is_active']) ? 1 : 0;
        $_SESSION[self::OLD_INPUT_KEY] = $payload;
    }

    /**
     * @return array<string, mixed>
     */
    private function pullOldInput(): array
    {
        $data = $_SESSION[self::OLD_INPUT_KEY] ?? [];
        unset($_SESSION[self::OLD_INPUT_KEY]);

        return is_array($data) ? $data : [];
    }

    private function clearOldInput(): void
    {
        unset($_SESSION[self::OLD_INPUT_KEY]);
    }

    private function redirect(string $path): void
    {
        header('Location: ' . app_url($path));
        exit;
    }
}
