<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\AuthService;
use App\Services\SsoTokenService;
use RuntimeException;

final class AuthController extends Controller
{
    public function __construct(
        private readonly AuthService $authService,
        private readonly SsoTokenService $ssoTokenService
    ) {
    }

    public function showLogin(): string
    {
        if ($this->authService->check()) {
            $this->redirect($this->sanitizeRedirect($_GET['redirect'] ?? '/dashboard'));
        }

        $errors = $this->pullFlash('auth_errors', []);
        $old = $this->pullFlash('auth_old', []);
        $redirect = $this->sanitizeRedirect($_GET['redirect'] ?? ($old['redirect'] ?? '/dashboard'));

        return $this->render('auth/login', [
            'errors' => $errors,
            'old' => array_merge(['email' => '', 'redirect' => $redirect], $old),
        ]);
    }

    public function handleLogin(): void
    {
        $errors = [];
        try {
            csrf_validate();
        } catch (RuntimeException $exception) {
            $errors[] = $exception->getMessage();
        }

        $blockedSeconds = $this->loginBlockedSeconds();
        if ($blockedSeconds !== null) {
            $errors[] = 'Muitas tentativas de login. Aguarde ' . $blockedSeconds . 's e tente novamente.';
        }

        $email = trim((string) ($_POST['email'] ?? ''));
        $password = (string) ($_POST['password'] ?? '');
        $redirect = $this->sanitizeRedirect($_POST['redirect'] ?? $_GET['redirect'] ?? '/dashboard');

        if ($email === '' || $password === '') {
            $errors[] = 'Informe email e senha.';
        }

        $clientIp = $this->clientIp();

        if ($errors === [] && !$this->authService->attemptLogin($email, $password)) {
            $this->incrementLoginAttempts();
            app_log('WARNING', 'Login falhou', ['email' => $email, 'ip' => $clientIp]);
            audit_log('auth', 'login_failed', null, ['email' => $email, 'ip' => $clientIp]);
            $errors[] = 'Credenciais invalidas ou usuario inativo.';
        }

        if ($errors !== []) {
            $this->flashBack($errors, ['email' => $email, 'redirect' => $redirect]);
            header('Location: ' . app_url('login'));
            exit;
        }

        $this->resetLoginAttempts();
        $authUser = $this->authService->user();
        app_log('INFO', 'Login efetuado', ['email' => $authUser['email'] ?? $email, 'ip' => $clientIp]);
        audit_log('auth', 'login_success', $authUser['id'] ?? null, ['ip' => $clientIp]);
        $this->redirect($redirect);
    }

    public function logout(): void
    {
        $this->authService->logout();
        header('Location: ' . app_url('login'));
        exit;
    }

    public function handleSso(): void
    {
        $redirect = $this->sanitizeRedirect($_GET['redirect'] ?? '/dashboard');
        $token = (string) ($_GET['token'] ?? '');

        try {
            $payload = $this->ssoTokenService->validateToken($token);
            if (!$this->authService->loginByEmail($payload['email'])) {
                throw new RuntimeException('Usuario nao encontrado ou inativo.');
            }

            $user = $this->authService->user();
            app_log('INFO', 'Login via SSO', ['email' => $payload['email'], 'ip' => $this->clientIp()]);
            audit_log('auth', 'sso_login', $user['id'] ?? null, ['email' => $payload['email']]);
            $this->redirect($redirect);
        } catch (RuntimeException $exception) {
            app_log('WARNING', 'SSO falhou', ['error' => $exception->getMessage(), 'ip' => $this->clientIp()]);
            $this->flashBack([$exception->getMessage()], ['redirect' => $redirect]);
            header('Location: ' . app_url('login'));
            exit;
        }
    }

    private function flashBack(array $errors, array $old = []): void
    {
        $_SESSION['auth_errors'] = $errors;
        $_SESSION['auth_old'] = $old;
    }

    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }

    private function sanitizeRedirect(string $target): string
    {
        $target = trim($target);
        if ($target === '') {
            return '/dashboard';
        }

        $path = parse_url($target, PHP_URL_PATH) ?? '';
        $query = parse_url($target, PHP_URL_QUERY);

        if ($path === '' || !str_starts_with($path, '/')) {
            $path = '/dashboard';
        }

        if ($path === '/login' || $path === '/logout') {
            $path = '/dashboard';
        }

        return $query ? $path . '?' . $query : $path;
    }

    private function redirect(string $path): void
    {
        header('Location: ' . app_url($path));
        exit;
    }

    private function loginRateState(): array
    {
        $state = $_SESSION['login_rate'] ?? [
            'attempts' => 0,
            'window_start' => time(),
            'blocked_until' => 0,
        ];

        $now = time();
        if ($now - ($state['window_start'] ?? 0) > 60) {
            $state['attempts'] = 0;
            $state['window_start'] = $now;
        }

        $_SESSION['login_rate'] = $state;

        return $state;
    }

    private function loginBlockedSeconds(): ?int
    {
        $state = $this->loginRateState();
        $blockedUntil = (int) ($state['blocked_until'] ?? 0);

        if ($blockedUntil > time()) {
            return $blockedUntil - time();
        }

        return null;
    }

    private function incrementLoginAttempts(): void
    {
        $state = $this->loginRateState();
        $state['attempts'] = (int) ($state['attempts'] ?? 0) + 1;

        if ($state['attempts'] >= 5) {
            $state['blocked_until'] = time() + 60;
            $state['attempts'] = 0;
            $state['window_start'] = time();
        }

        $_SESSION['login_rate'] = $state;
    }

    private function resetLoginAttempts(): void
    {
        $_SESSION['login_rate'] = [
            'attempts' => 0,
            'window_start' => time(),
            'blocked_until' => 0,
        ];
    }

    private function clientIp(): string
    {
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
}
