<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Repositories\ReportRepository;
use App\Services\AuthService;
use DateInterval;
use DateTimeImmutable;
use RuntimeException;

final class ReportController extends Controller
{
    public function __construct(
        private readonly AuthService $authService,
        private readonly ReportRepository $reports
    ) {
    }

    public function index(): string
    {
        $this->requireGestor();
        [$fromDate, $toDate, $filters] = $this->resolvePeriod();

        $categoryPriority = $this->reports->categoryPriorityMatrix($fromDate, $toDate);
        $typeBreakdown = $this->reports->typeSubcategoryBreakdown($fromDate, $toDate);
        $averages = $this->reports->averageTimes($fromDate, $toDate);
        $sla = $this->reports->slaCompliance($fromDate, $toDate);
        $rankingAssignees = $this->reports->rankingByAssignee($fromDate, $toDate, 6);
        $rankingCustomers = $this->reports->rankingByCustomer($fromDate, $toDate, 6);

        $totalsByMain = $this->aggregateTotals($typeBreakdown);

        return $this->render('reports/index', [
            'filters' => $filters,
            'categoryPriority' => $categoryPriority,
            'typeBreakdown' => $typeBreakdown,
            'totalsByMain' => $totalsByMain,
            'averages' => $averages,
            'sla' => $sla,
            'rankingAssignees' => $rankingAssignees,
            'rankingCustomers' => $rankingCustomers,
            'pageTitle' => 'Dashboard Executivo',
        ]);
    }

    public function export(): void
    {
        $this->requireGestor();
        [$fromDate, $toDate,] = $this->resolvePeriod();

        $metric = $_GET['metric'] ?? '';
        $data = [];
        $headers = [];
        $filename = 'relatorio.csv';
        $formatter = static fn (array $row): array => array_values($row);

        switch ($metric) {
            case 'category_priority':
                $data = $this->reports->categoryPriorityMatrix($fromDate, $toDate);
                $headers = ['Categoria', 'Prioridade', 'Total'];
                $filename = 'categorias-prioridades.csv';
                 $formatter = static fn (array $row): array => [
                    $row['category_name'] ?? '',
                    $row['priority_name'] ?? '',
                    (int) ($row['total'] ?? 0),
                 ];
                break;
            case 'type_subcategory':
                $data = $this->reports->typeSubcategoryBreakdown($fromDate, $toDate);
                $headers = ['Tipo', 'Subcategoria', 'Total'];
                $filename = 'tipo-subcategoria.csv';
                $formatter = static fn (array $row): array => [
                    $row['main_category'] ?? '',
                    $row['subcategory'] ?? '',
                    (int) ($row['total'] ?? 0),
                ];
                break;
            case 'ranking_assignees':
                $data = $this->reports->rankingByAssignee($fromDate, $toDate, 50);
                $headers = ['Responsável', 'Chamados', 'Média Resolução (min)'];
                $filename = 'ranking-responsaveis.csv';
                $formatter = static fn (array $row): array => [
                    $row['assignee_name'] ?? '',
                    (int) ($row['total_tickets'] ?? 0),
                    isset($row['avg_resolution']) && $row['avg_resolution'] !== null
                        ? number_format((float) $row['avg_resolution'], 2, ',', '')
                        : '',
                ];
                break;
            case 'ranking_customers':
                $data = $this->reports->rankingByCustomer($fromDate, $toDate, 50);
                $headers = ['Cliente/Obra', 'Chamados', 'Média Resolução (min)'];
                $filename = 'ranking-clientes.csv';
                $formatter = static fn (array $row): array => [
                    $row['customer_name'] ?? '',
                    (int) ($row['total_tickets'] ?? 0),
                    isset($row['avg_resolution']) && $row['avg_resolution'] !== null
                        ? number_format((float) $row['avg_resolution'], 2, ',', '')
                        : '',
                ];
                break;
            case 'sla':
                $data = [$this->reports->slaCompliance($fromDate, $toDate)];
                $headers = ['SLA Atendido', 'SLA Não Atendido'];
                $filename = 'sla.csv';
                $formatter = static fn (array $row): array => [
                    (int) ($row['met'] ?? 0),
                    (int) ($row['breached'] ?? 0),
                ];
                break;
            case 'averages':
                $data = [$this->reports->averageTimes($fromDate, $toDate)];
                $headers = ['Média Primeira Resposta (min)', 'Média Resolução (min)'];
                $filename = 'tempos-medios.csv';
                $formatter = static fn (array $row): array => [
                    isset($row['avg_response']) && $row['avg_response'] !== null
                        ? number_format((float) $row['avg_response'], 2, ',', '')
                        : '',
                    isset($row['avg_resolution']) && $row['avg_resolution'] !== null
                        ? number_format((float) $row['avg_resolution'], 2, ',', '')
                        : '',
                ];
                break;
            default:
                http_response_code(400);
                echo 'Métrica inválida para exportação.';
                return;
        }

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        $output = fopen('php://output', 'w');

        fputcsv($output, $headers);

        foreach ($data as $row) {
            fputcsv($output, $formatter($row));
        }

        fclose($output);
        exit;
    }

    /**
     * @return array{0:?string,1:?string,2:array<string,string>}
     */
    private function resolvePeriod(): array
    {
        $fromInput = $_GET['from_date'] ?? null;
        $toInput = $_GET['to_date'] ?? null;

        $fromDate = $this->sanitizeDate($fromInput);
        $toDate = $this->sanitizeDate($toInput);

        if ($fromDate === null && $toDate === null) {
            $toDate = (new DateTimeImmutable('today'))->format('Y-m-d');
            $fromDate = (new DateTimeImmutable($toDate))
                ->sub(new DateInterval('P30D'))
                ->format('Y-m-d');
        }

        if ($fromDate !== null && $toDate !== null) {
            if ($fromDate > $toDate) {
                [$fromDate, $toDate] = [$toDate, $fromDate];
            }
        }

        return [
            $fromDate,
            $toDate,
            [
                'from_date' => $fromDate ?? '',
                'to_date' => $toDate ?? '',
            ],
        ];
    }

    private function sanitizeDate(?string $value): ?string
    {
        if ($value === null || trim($value) === '') {
            return null;
        }

        $value = trim($value);
        $timestamp = strtotime($value);
        if ($timestamp === false) {
            return null;
        }

        return date('Y-m-d', $timestamp);
    }

    private function requireGestor(): void
    {
        $user = $this->authService->user();
        if ($user === null || ($user['role_slug'] ?? '') !== 'gestor') {
            throw new RuntimeException('Acesso restrito aos gestores.');
        }
    }

    /**
     * @param array<int, array<string, mixed>> $breakdown
     * @return array<string, int>
     */
    private function aggregateTotals(array $breakdown): array
    {
        $totals = [];
        foreach ($breakdown as $row) {
            $type = $row['main_category'] ?? 'N/A';
            $totals[$type] = ($totals[$type] ?? 0) + (int) ($row['total'] ?? 0);
        }

        return $totals;
    }
}
