<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class AttachmentRepository
{
    /**
     * @param array<string, mixed> $data
     */
    public function create(array $data): int
    {
        $columns = array_keys($data);
        $placeholders = array_map(
            static fn (string $column): string => ':' . $column,
            $columns
        );

        $sql = sprintf(
            'INSERT INTO ticket_attachments (%s) VALUES (%s)',
            implode(', ', $columns),
            implode(', ', $placeholders)
        );

        Database::run($sql, $data);

        return (int) Database::connection()->lastInsertId();
    }

    public function findById(int $id): ?array
    {
        $sql = <<<SQL
            SELECT
                a.id,
                a.ticket_id,
                a.comment_id,
                a.filename,
                a.path,
                a.mime_type,
                a.file_size,
                a.created_at,
                c.is_internal AS comment_is_internal
            FROM ticket_attachments a
            LEFT JOIN ticket_comments c ON c.id = a.comment_id
            WHERE a.id = :id
            LIMIT 1
        SQL;

        $statement = Database::run($sql, ['id' => $id]);
        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? null : $row;
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function listByTicket(int $ticketId): array
    {
        $sql = <<<SQL
            SELECT
                id,
                ticket_id,
                comment_id,
                filename,
                path,
                mime_type,
                file_size,
                created_at
            FROM ticket_attachments
            WHERE ticket_id = :ticket_id
            ORDER BY created_at DESC
        SQL;

        $statement = Database::run($sql, ['ticket_id' => $ticketId]);

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @param array<int, int> $commentIds
     * @return array<int, array<string, mixed>>
     */
    public function listByCommentIds(array $commentIds): array
    {
        if ($commentIds === []) {
            return [];
        }

        $placeholders = [];
        $parameters = [];

        foreach (array_values($commentIds) as $index => $commentId) {
            $key = ':id' . $index;
            $placeholders[] = $key;
            $parameters[$key] = (int) $commentId;
        }

        $sql = sprintf(
            'SELECT id, ticket_id, comment_id, filename, path, mime_type, file_size, created_at
             FROM ticket_attachments
             WHERE comment_id IN (%s)
             ORDER BY created_at ASC',
            implode(', ', $placeholders)
        );

        $statement = Database::run($sql, $parameters);

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
}
