<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class NotificationOutboxRepository
{
    /**
     * @param array<string, mixed> $data
     */
    public function insert(array $data): void
    {
        $columns = array_keys($data);
        $placeholders = array_map(static fn (string $column): string => ':' . $column, $columns);
        $sql = sprintf(
            'INSERT INTO notification_outbox (%s) VALUES (%s)',
            implode(', ', $columns),
            implode(', ', $placeholders)
        );

        Database::run($sql, $data);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function pending(int $limit = 25): array
    {
        $sql = <<<SQL
            SELECT *
            FROM notification_outbox
            WHERE status = 'pending'
            ORDER BY id ASC
            LIMIT :limit
        SQL;

        $statement = Database::connection()->prepare($sql);
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->execute();

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    public function markSent(int $id): void
    {
        $sql = <<<SQL
            UPDATE notification_outbox
            SET status = 'sent', sent_at = NOW(), attempts = attempts + 1, last_error = NULL
            WHERE id = :id
        SQL;

        Database::run($sql, ['id' => $id]);
    }

    public function markError(int $id, string $message): void
    {
        $sql = <<<SQL
            UPDATE notification_outbox
            SET status = 'error', attempts = attempts + 1, last_error = :last_error
            WHERE id = :id
        SQL;

        Database::run($sql, [
            'id' => $id,
            'last_error' => mb_strimwidth($message, 0, 250),
        ]);
    }

    public function pendingCount(): int
    {
        $statement = Database::run(
            "SELECT COUNT(*) AS total FROM notification_outbox WHERE status = 'pending'"
        );

        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? 0 : (int) $row['total'];
    }
}
