<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;
use PDOException;

final class QueueRepository
{
    /**
     * @return array<int, array<string, mixed>>
     */
    public function all(): array
    {
        $sql = 'SELECT id, name, slug, description, sla_response_minutes, sla_resolution_minutes, is_active FROM queues ORDER BY name ASC';
        $statement = Database::run($sql);

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @param array<string, mixed> $data
     */
    public function create(array $data): int
    {
        $sql = <<<SQL
            INSERT INTO queues (slug, name, description, sla_response_minutes, sla_resolution_minutes, is_active)
            VALUES (:slug, :name, :description, :sla_response_minutes, :sla_resolution_minutes, :is_active)
        SQL;

        Database::run($sql, $data);
        return (int) Database::connection()->lastInsertId();
    }

    /**
     * @return array{rows: array<int, array<string, mixed>>, total: int}
     */
    public function paginate(?string $search, ?string $status, int $limit, int $offset): array
    {
        $conditions = ['1=1'];
        $params = [];

        if ($search !== null && $search !== '') {
            $conditions[] = '(q.name LIKE :search OR q.slug LIKE :search)';
            $params['search'] = '%' . $search . '%';
        }

        if ($status === 'active') {
            $conditions[] = 'q.is_active = 1';
        } elseif ($status === 'inactive') {
            $conditions[] = 'q.is_active = 0';
        }

        $where = 'WHERE ' . implode(' AND ', $conditions);

        $countSql = 'SELECT COUNT(*) FROM queues q ' . $where;
        $countStmt = Database::prepare($countSql);
        foreach ($params as $key => $value) {
            $countStmt->bindValue(':' . $key, $value);
        }
        $countStmt->execute();
        $total = (int) $countStmt->fetchColumn();

        $sql = <<<SQL
            SELECT q.*
            FROM queues q
            $where
            ORDER BY q.name ASC, q.id ASC
            LIMIT :limit OFFSET :offset
        SQL;

        $statement = Database::connection()->prepare($sql);
        foreach ($params as $key => $value) {
            $statement->bindValue(':' . $key, $value);
        }
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->bindValue(':offset', $offset, PDO::PARAM_INT);
        $statement->execute();

        return [
            'rows' => $statement->fetchAll(PDO::FETCH_ASSOC) ?: [],
            'total' => $total,
        ];
    }

    public function findById(int $id): ?array
    {
        $sql = 'SELECT * FROM queues WHERE id = :id LIMIT 1';
        $statement = Database::run($sql, ['id' => $id]);
        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? null : $row;
    }

    /**
     * @param array<string, mixed> $data
     */
    public function update(int $id, array $data): bool
    {
        $fields = [];
        foreach ($data as $column => $_value) {
            $fields[] = sprintf('%s = :%s', $column, $column);
        }
        $fields[] = 'updated_at = CURRENT_TIMESTAMP';

        $sql = sprintf(
            'UPDATE queues SET %s WHERE id = :id',
            implode(', ', $fields)
        );
        $data['id'] = $id;

        $statement = Database::run($sql, $data);
        return $statement->rowCount() > 0;
    }

    public function delete(int $id): bool
    {
        try {
            $statement = Database::run('DELETE FROM queues WHERE id = :id', ['id' => $id]);
            return $statement->rowCount() > 0;
        } catch (PDOException $exception) {
            throw $exception;
        }
    }

    public function toggleStatus(int $id, int $status): bool
    {
        $statement = Database::run(
            'UPDATE queues SET is_active = :is_active, updated_at = NOW() WHERE id = :id',
            [
                'id' => $id,
                'is_active' => $status,
            ]
        );

        return $statement->rowCount() > 0;
    }

    public function nameExists(string $name, ?int $ignoreId = null): bool
    {
        $sql = 'SELECT 1 FROM queues WHERE name = :name';
        $params = ['name' => $name];

        if ($ignoreId !== null) {
            $sql .= ' AND id <> :ignore_id';
            $params['ignore_id'] = $ignoreId;
        }

        $sql .= ' LIMIT 1';

        $statement = Database::run($sql, $params);
        return $statement->fetchColumn() !== false;
    }

    public function slugExists(string $slug, ?int $ignoreId = null): bool
    {
        $sql = 'SELECT 1 FROM queues WHERE slug = :slug';
        $params = ['slug' => $slug];

        if ($ignoreId !== null) {
            $sql .= ' AND id <> :ignore_id';
            $params['ignore_id'] = $ignoreId;
        }

        $sql .= ' LIMIT 1';

        $statement = Database::run($sql, $params);
        return $statement->fetchColumn() !== false;
    }
}
