<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class SlaPolicyRepository
{
    /**
     * @return array<int, array<string, mixed>>
     */
    public function all(): array
    {
        $sql = <<<SQL
            SELECT
                p.id,
                p.category_id,
                c.name AS category_name,
                p.priority_id,
                pr.name AS priority_name,
                p.client_type,
                p.response_minutes,
                p.resolution_minutes,
                p.sort_order,
                p.is_active
            FROM sla_policies p
            LEFT JOIN categories c ON c.id = p.category_id
            LEFT JOIN priorities pr ON pr.id = p.priority_id
            ORDER BY c.name ASC, pr.sort_order ASC, p.client_type ASC
        SQL;

        $statement = Database::run($sql);
        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @param array<string, mixed> $data
     */
    public function create(array $data): int
    {
        $sql = <<<SQL
            INSERT INTO sla_policies (category_id, priority_id, client_type, response_minutes, resolution_minutes, sort_order, is_active)
            VALUES (:category_id, :priority_id, :client_type, :response_minutes, :resolution_minutes, :sort_order, :is_active)
        SQL;

        Database::run($sql, $data);
        return (int) Database::connection()->lastInsertId();
    }

    public function findPolicy(?int $categoryId, int $priorityId, string $clientType): ?array
    {
        $sql = <<<SQL
            SELECT
                id,
                category_id,
                priority_id,
                client_type,
                response_minutes,
                resolution_minutes
            FROM sla_policies
            WHERE is_active = 1
              AND (category_id IS NULL OR category_id = :category_id)
              AND (priority_id IS NULL OR priority_id = :priority_id)
              AND (client_type IS NULL OR client_type = :client_type)
            ORDER BY
                (CASE WHEN category_id IS NULL THEN 0 ELSE 1 END
                 + CASE WHEN priority_id IS NULL THEN 0 ELSE 1 END
                 + CASE WHEN client_type IS NULL THEN 0 ELSE 1 END) DESC,
                sort_order ASC,
                id ASC
            LIMIT 1
        SQL;

        $statement = Database::connection()->prepare($sql);
        $statement->bindValue(':category_id', $categoryId, $categoryId === null ? PDO::PARAM_NULL : PDO::PARAM_INT);
        $statement->bindValue(':priority_id', $priorityId, PDO::PARAM_INT);
        $statement->bindValue(':client_type', $clientType);
        $statement->execute();

        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? null : $row;
    }
}
