<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class ThemeRepository
{
    public function findByContext(string $context): ?array
    {
        $sql = 'SELECT * FROM settings_theme WHERE context = :context LIMIT 1';
        $statement = Database::run($sql, ['context' => $context]);
        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? null : $row;
    }

    public function findDefault(): ?array
    {
        $sql = 'SELECT * FROM settings_theme ORDER BY id ASC LIMIT 1';
        $statement = Database::run($sql);
        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? null : $row;
    }

    /**
     * @param array<string, string|null> $data
     */
    public function save(string $context, array $data): void
    {
        $sql = <<<SQL
            INSERT INTO settings_theme (
                context,
                primary_color,
                secondary_color,
                dark_color,
                accent_color,
                logo_text,
                logo_path
            ) VALUES (
                :context,
                :primary_color,
                :secondary_color,
                :dark_color,
                :accent_color,
                :logo_text,
                :logo_path
            )
            ON DUPLICATE KEY UPDATE
                primary_color = VALUES(primary_color),
                secondary_color = VALUES(secondary_color),
                dark_color = VALUES(dark_color),
                accent_color = VALUES(accent_color),
                logo_text = VALUES(logo_text),
                logo_path = VALUES(logo_path),
                updated_at = CURRENT_TIMESTAMP
        SQL;

        Database::run($sql, array_merge($data, ['context' => $context]));
    }

    public function updateLogoPath(string $context, ?string $path): void
    {
        $existing = $this->findByContext($context);
        if ($existing === null) {
            $this->save($context, [
                'primary_color' => '#ff7a18',
                'secondary_color' => '#6c63ff',
                'dark_color' => '#1f1f1f',
                'accent_color' => '#37474f',
                'logo_text' => 'HelpDesk',
                'logo_path' => $path,
            ]);
            return;
        }

        $sql = 'UPDATE settings_theme SET logo_path = :logo_path, updated_at = CURRENT_TIMESTAMP WHERE context = :context';
        Database::run($sql, [
            'logo_path' => $path,
            'context' => $context,
        ]);
    }
}
