<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\AttachmentRepository;
use RuntimeException;

final class AttachmentService
{
    private const MAX_FILE_SIZE = 20_971_520; // 20 MB

    /**
     * @var string[]
     */
    private array $allowedMimeTypes = [
        'image/jpeg',
        'image/png',
        'image/gif',
        'image/webp',
        'application/pdf',
        'video/mp4',
        'video/quicktime',
        'video/x-msvideo',
        'video/x-matroska',
    ];

    public function __construct(
        private readonly AttachmentRepository $repository
    ) {
    }

    /**
     * @param array<string, mixed>|null $files
     * @return array<int, array<string, mixed>>
     */
    public function storeTicketFiles(int $ticketId, int $commentId, ?array $files): array
    {
        $uploads = $this->normalizeUploads($files);
        if ($uploads === []) {
            return [];
        }

        $saved = [];

        foreach ($uploads as $upload) {
            $error = (int) ($upload['error'] ?? UPLOAD_ERR_NO_FILE);
            if ($error === UPLOAD_ERR_NO_FILE) {
                continue;
            }

            if ($error !== UPLOAD_ERR_OK) {
                throw new RuntimeException('Falha ao enviar arquivo: ' . $this->translateUploadError($error));
            }

            $size = (int) ($upload['size'] ?? 0);
            if ($size <= 0) {
                throw new RuntimeException('Arquivo vazio ou inválido.');
            }

            if ($size > self::MAX_FILE_SIZE) {
                throw new RuntimeException('Arquivo excede o limite de 20MB.');
            }

            $tmpPath = (string) ($upload['tmp_name'] ?? '');
            if ($tmpPath === '' || !is_uploaded_file($tmpPath)) {
                throw new RuntimeException('Upload inválido.');
            }

            $mime = $this->detectMimeType($tmpPath, (string) ($upload['type'] ?? ''));
            if (!$this->isAllowedMime($mime)) {
                throw new RuntimeException('Tipo de arquivo não permitido: ' . $mime);
            }

            $originalName = $this->sanitizeFilename((string) ($upload['name'] ?? 'arquivo'));
            $extension = $this->extensionFromNameOrMime($originalName, $mime);
            $relativeDir = sprintf('storage/uploads/%s/%s', date('Y'), date('m'));
            $absoluteDir = BASE_PATH . '/' . $relativeDir;
            $this->ensureDirectory($absoluteDir);

            $uniqueName = $this->generateUniqueName($ticketId, $extension);
            $targetPath = $absoluteDir . '/' . $uniqueName;

            if (!move_uploaded_file($tmpPath, $targetPath)) {
                throw new RuntimeException('Não foi possível salvar o arquivo enviado.');
            }

            $relativePath = $relativeDir . '/' . $uniqueName;
            $row = [
                'ticket_id' => $ticketId,
                'comment_id' => $commentId,
                'filename' => $originalName,
                'path' => $relativePath,
                'mime_type' => $mime,
                'file_size' => $size,
            ];

            $id = $this->repository->create($row);
            $saved[] = array_merge(['id' => $id], $row);
        }

        return $saved;
    }

    public function hasUploads(?array $files): bool
    {
        $uploads = $this->normalizeUploads($files);

        foreach ($uploads as $upload) {
            $error = (int) ($upload['error'] ?? UPLOAD_ERR_NO_FILE);
            if ($error !== UPLOAD_ERR_NO_FILE) {
                return true;
            }
        }

        return false;
    }

    public function find(int $attachmentId): ?array
    {
        return $this->repository->findById($attachmentId);
    }

    /**
     * @param array<int, int> $commentIds
     * @return array<int, array<int, array<string, mixed>>>
     */
    public function attachmentsGroupedByComment(array $commentIds): array
    {
        if ($commentIds === []) {
            return [];
        }

        $rows = $this->repository->listByCommentIds($commentIds);
        $grouped = [];

        foreach ($rows as $row) {
            $commentId = (int) ($row['comment_id'] ?? 0);
            if ($commentId === 0) {
                continue;
            }

            $row['is_previewable'] = $this->isPreviewable((string) ($row['mime_type'] ?? ''));
            $row['size_label'] = $this->formatBytes((int) ($row['file_size'] ?? 0));
            $grouped[$commentId][] = $row;
        }

        return $grouped;
    }

    public function stream(array $attachment, bool $inline = false): void
    {
        $relativePath = (string) ($attachment['path'] ?? '');
        $filePath = BASE_PATH . '/' . ltrim($relativePath, '/');

        if (!is_file($filePath)) {
            throw new RuntimeException('Arquivo não encontrado.');
        }

        $mime = (string) ($attachment['mime_type'] ?? 'application/octet-stream');
        $filename = (string) ($attachment['filename'] ?? basename($filePath));
        $disposition = $inline ? 'inline' : 'attachment';
        $filesize = (int) filesize($filePath);

        header('Content-Type: ' . $mime);
        header('Content-Length: ' . $filesize);
        header('Content-Disposition: ' . $disposition . '; filename="' . $this->escapeHeaderValue($filename) . '"');
        header('X-Content-Type-Options: nosniff');

        $handle = fopen($filePath, 'rb');
        if ($handle === false) {
            throw new RuntimeException('Não foi possível abrir o arquivo.');
        }

        while (!feof($handle)) {
            $chunk = fread($handle, 1048576);
            if ($chunk === false) {
                break;
            }

            echo $chunk;
        }

        fclose($handle);
    }

    /**
     * @param array<string, mixed>|null $files
     * @return array<int, array<string, mixed>>
     */
    private function normalizeUploads(?array $files): array
    {
        if ($files === null || $files === []) {
            return [];
        }

        if (!is_array($files['name'] ?? null)) {
            return [$files];
        }

        $normalized = [];
        $names = $files['name'];
        $types = $files['type'] ?? [];
        $tmpNames = $files['tmp_name'] ?? [];
        $errors = $files['error'] ?? [];
        $sizes = $files['size'] ?? [];

        foreach ($names as $index => $name) {
            $normalized[] = [
                'name' => $name,
                'type' => $types[$index] ?? '',
                'tmp_name' => $tmpNames[$index] ?? '',
                'error' => $errors[$index] ?? UPLOAD_ERR_NO_FILE,
                'size' => $sizes[$index] ?? 0,
            ];
        }

        return $normalized;
    }

    private function translateUploadError(int $error): string
    {
        return match ($error) {
            UPLOAD_ERR_INI_SIZE,
            UPLOAD_ERR_FORM_SIZE => 'Arquivo excede o limite permitido.',
            UPLOAD_ERR_PARTIAL => 'Upload de arquivo incompleto.',
            UPLOAD_ERR_NO_TMP_DIR => 'Diretório temporário ausente.',
            UPLOAD_ERR_CANT_WRITE => 'Falha ao gravar o arquivo.',
            UPLOAD_ERR_EXTENSION => 'Upload bloqueado por extensão.',
            default => 'Falha desconhecida no upload.',
        };
    }

    private function detectMimeType(string $tmpPath, string $fallback): string
    {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        if ($finfo !== false) {
            $detected = finfo_file($finfo, $tmpPath);
            finfo_close($finfo);

            if (is_string($detected) && $detected !== '') {
                return $detected;
            }
        }

        return $fallback !== '' ? $fallback : 'application/octet-stream';
    }

    private function isAllowedMime(string $mime): bool
    {
        return in_array(strtolower($mime), $this->allowedMimeTypes, true);
    }

    private function isPreviewable(string $mime): bool
    {
        return str_starts_with(strtolower($mime), 'image/');
    }

    private function sanitizeFilename(string $filename): string
    {
        $basename = basename($filename);
        $clean = preg_replace('/[^A-Za-z0-9_\-\.]+/', '_', $basename) ?? 'arquivo';

        return trim($clean, '_') !== '' ? $clean : 'arquivo';
    }

    private function extensionFromNameOrMime(string $filename, string $mime): string
    {
        $extension = strtolower((string) pathinfo($filename, PATHINFO_EXTENSION));
        if ($extension !== '') {
            return $extension;
        }

        return match (strtolower($mime)) {
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'application/pdf' => 'pdf',
            'video/mp4' => 'mp4',
            'video/quicktime' => 'mov',
            'video/x-msvideo' => 'avi',
            'video/x-matroska' => 'mkv',
            default => 'dat',
        };
    }

    private function ensureDirectory(string $path): void
    {
        if (!is_dir($path)) {
            mkdir($path, 0775, true);
        }
    }

    private function generateUniqueName(int $ticketId, string $extension): string
    {
        $random = bin2hex(random_bytes(5));
        $suffix = $extension !== '' ? '.' . $extension : '';

        return sprintf('ticket-%d-%s-%s%s', $ticketId, date('YmdHis'), $random, $suffix);
    }

    private function formatBytes(int $bytes): string
    {
        if ($bytes < 1024) {
            return $bytes . ' B';
        }

        $units = ['KB', 'MB', 'GB', 'TB'];
        $value = $bytes / 1024;
        $unitIndex = 0;

        while ($value >= 1024 && $unitIndex < count($units) - 1) {
            $value /= 1024;
            $unitIndex++;
        }

        return sprintf('%.1f %s', $value, $units[$unitIndex]);
    }

    private function escapeHeaderValue(string $value): string
    {
        return str_replace('"', '', $value);
    }
}
