<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\UserRepository;

final class AuthService
{
    private const SESSION_KEY = 'auth_user';

    public function __construct(
        private readonly UserRepository $users
    ) {
    }

    public function attemptLogin(string $email, string $password): bool
    {
        $email = strtolower(trim($email));
        if ($email === '' || $password === '') {
            return false;
        }

        $user = $this->users->findByEmail($email);
        if ($user === null || (int) $user['is_active'] !== 1) {
            return false;
        }

        if (!password_verify($password, $user['password_hash'])) {
            return false;
        }

        $this->storeUserInSession($user);
        $this->users->updateLastLogin((int) $user['id']);

        return true;
    }

    public function loginUserId(int $id): bool
    {
        $user = $this->users->findById($id);
        if ($user === null || (int) $user['is_active'] !== 1) {
            return false;
        }

        $this->storeUserInSession($user);
        $this->users->updateLastLogin((int) $user['id']);

        return true;
    }

    public function loginUserRecord(array $user): bool
    {
        if (!isset($user['id']) || (int) ($user['is_active'] ?? 0) !== 1) {
            return false;
        }

        $this->storeUserInSession($user);
        $this->users->updateLastLogin((int) $user['id']);

        return true;
    }

    public function loginByEmail(string $email): bool
    {
        $email = strtolower(trim($email));
        if ($email === '') {
            return false;
        }

        $user = $this->users->findByEmail($email);
        if ($user === null) {
            return false;
        }

        return $this->loginUserRecord($user);
    }

    public function logout(): void
    {
        unset($_SESSION[self::SESSION_KEY]);
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_regenerate_id(true);
        }
    }

    public function user(): ?array
    {
        return $_SESSION[self::SESSION_KEY] ?? null;
    }

    public function check(): bool
    {
        return $this->user() !== null;
    }

    public function hasRole(array $roles): bool
    {
        if ($roles === []) {
            return true;
        }

        $user = $this->user();
        if ($user === null) {
            return false;
        }

        return in_array($user['role_slug'] ?? '', $roles, true);
    }

    private function storeUserInSession(array $user): void
    {
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_regenerate_id(true);
        }

        $_SESSION[self::SESSION_KEY] = [
            'id' => (int) $user['id'],
            'name' => $user['name'],
            'email' => $user['email'],
            'role_slug' => $user['role_slug'],
            'role_name' => $user['role_name'],
            'queue_id' => isset($user['queue_id']) ? (int) $user['queue_id'] : null,
        ];
    }
}
