<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\AdminAuditRepository;
use App\Repositories\CategoryRepository;
use App\Repositories\TicketFieldRepository;
use RuntimeException;

final class FieldService
{
    public function __construct(
        private readonly TicketFieldRepository $fields,
        private readonly CategoryRepository $categories,
        private readonly AdminAuditRepository $audit
    ) {
    }

    /**
     * @return array{data: array<int, array<string, mixed>>, pagination: array<string, int>, filters: array<string, mixed>}
     */
    public function getPaginated(?string $search, ?string $status, int $page, int $perPage = 10): array
    {
        $search = $search !== null ? trim($search) : null;
        $status = in_array($status, ['active', 'inactive'], true) ? $status : null;
        $page = max(1, $page);

        $offset = ($page - 1) * $perPage;
        $result = $this->fields->paginate($search, $status, $perPage, $offset);

        $pages = (int) max(1, ceil($result['total'] / $perPage));
        if ($page > $pages && $result['total'] > 0) {
            $page = $pages;
            $offset = ($page - 1) * $perPage;
            $result = $this->fields->paginate($search, $status, $perPage, $offset);
        }

        return [
            'data' => $result['rows'],
            'pagination' => [
                'page' => $page,
                'pages' => $pages,
                'per_page' => $perPage,
                'total' => $result['total'],
            ],
            'filters' => [
                'search' => $search,
                'status' => $status ?? 'all',
            ],
        ];
    }

    public function findById(int $id): ?array
    {
        return $this->fields->findById($id);
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed>|null $actor
     */
    public function create(array $input, ?array $actor = null): int
    {
        $data = $this->validatePayload($input);
        $fieldId = $this->fields->create($data);
        $this->log($actor, 'create', $fieldId, [
            'category_id' => $data['category_id'],
            'name' => $data['name'],
            'type' => $data['type'],
        ]);

        return $fieldId;
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed>|null $actor
     */
    public function update(int $id, array $input, ?array $actor = null): void
    {
        $record = $this->findById($id);
        if ($record === null) {
            throw new RuntimeException('Campo não encontrado.');
        }

        $data = $this->validatePayload($input, $id);
        if (!$this->fields->update($id, $data)) {
            throw new RuntimeException('Não foi possível atualizar o campo informado.');
        }

        $this->log($actor, 'update', $id, [
            'category_id' => $data['category_id'],
            'name' => $data['name'],
            'type' => $data['type'],
        ]);
    }

    public function delete(int $id, ?array $actor = null): void
    {
        if (!$this->fields->delete($id)) {
            throw new RuntimeException('Campo não encontrado ou já removido.');
        }

        $this->log($actor, 'delete', $id);
    }

    public function toggleStatus(int $id, bool $activate, ?array $actor = null): void
    {
        if (!$this->fields->toggleStatus($id, $activate ? 1 : 0)) {
            throw new RuntimeException('Não foi possível alterar o status do campo.');
        }

        $this->log($actor, 'toggle', $id, ['is_active' => $activate ? 1 : 0]);
    }

    public function nameExists(int $categoryId, string $name, ?int $ignoreId = null): bool
    {
        return $this->fields->nameExists($categoryId, $name, $ignoreId);
    }

    /**
     * @param array<string, mixed> $input
     * @return array<string, mixed>
     */
    private function validatePayload(array $input, ?int $ignoreId = null): array
    {
        $categoryId = $this->requiredInt($input['category_id'] ?? null, 'Selecione a subcategoria.');
        if ($this->categories->find($categoryId) === null) {
            throw new RuntimeException('Subcategoria não encontrada.');
        }

        $label = trim((string) ($input['label'] ?? ''));
        if (mb_strlen($label) < 3) {
            throw new RuntimeException('O rótulo deve ter pelo menos 3 caracteres.');
        }

        $type = strtolower(trim((string) ($input['type'] ?? '')));
        $allowedTypes = ['text', 'textarea', 'select', 'number', 'boolean'];
        if (!in_array($type, $allowedTypes, true)) {
            throw new RuntimeException('Tipo de campo inválido.');
        }

        $nameSource = (string) ($input['name'] ?? $label);
        $name = $this->slugify($nameSource);
        if (mb_strlen($name) < 3) {
            throw new RuntimeException('Defina um identificador técnico com ao menos 3 caracteres.');
        }

        if ($this->nameExists($categoryId, $name, $ignoreId)) {
            throw new RuntimeException('Já existe um campo com este nome para a categoria selecionada.');
        }

        $options = null;
        if ($type === 'select') {
            $optionsText = trim((string) ($input['options'] ?? ''));
            if ($optionsText === '') {
                throw new RuntimeException('Informe opções para o campo select (uma por linha).');
            }
            $optionsArray = array_values(array_filter(array_map(
                'trim',
                preg_split('/[\r\n,]+/', $optionsText) ?: []
            )));
            if ($optionsArray === []) {
                throw new RuntimeException('Não há opções válidas para o campo select.');
            }
            $options = json_encode($optionsArray, JSON_UNESCAPED_UNICODE);
        }

        $sortOrder = isset($input['sort_order']) ? max(0, (int) $input['sort_order']) : 0;

        return [
            'category_id' => $categoryId,
            'label' => $label,
            'name' => $name,
            'type' => $type,
            'options' => $options,
            'is_required' => isset($input['is_required']) ? 1 : 0,
            'sort_order' => $sortOrder,
            'is_active' => isset($input['is_active']) ? 1 : 0,
        ];
    }

    private function slugify(string $value): string
    {
        $value = strtolower(trim($value));
        $value = preg_replace('/[^a-z0-9]+/i', '-', $value) ?? '';

        return trim($value, '-');
    }

    private function requiredInt(mixed $value, string $message): int
    {
        $numeric = $this->nullableInt($value);
        if ($numeric === null) {
            throw new RuntimeException($message);
        }

        return $numeric;
    }

    private function nullableInt(mixed $value): ?int
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return (int) $value;
        }

        return null;
    }

    /**
     * @param array<string, mixed>|null $actor
     * @param array<string, mixed> $payload
     */
    private function log(?array $actor, string $action, ?int $entityId, array $payload = []): void
    {
        $userId = isset($actor['id']) ? (int) $actor['id'] : null;
        \app_log('INFO', 'Field action', [
            'user_id' => $userId,
            'action' => $action,
            'entity_id' => $entityId,
        ]);

        $this->audit->log($userId, 'ticket_field', $action, $entityId, $payload);
    }
}
