<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\AdminAuditRepository;

final class LogService
{
    public function __construct(
        private readonly AdminAuditRepository $audits,
        private readonly string $applicationLogPath,
        private readonly string $pdoLogPath
    ) {
    }

    /**
     * @return array<string, mixed>
     */
    public function getApplicationLogs(?string $startDate, ?string $endDate, int $page, int $perPage = 50): array
    {
        return $this->buildFileReport($this->applicationLogPath, $startDate, $endDate, $page, $perPage);
    }

    /**
     * @return array<string, mixed>
     */
    public function getPdoLogs(?string $startDate, ?string $endDate, int $page, int $perPage = 50): array
    {
        return $this->buildFileReport($this->pdoLogPath, $startDate, $endDate, $page, $perPage);
    }

    /**
     * @return array<string, mixed>
     */
    public function getAuditLogs(?string $startDate, ?string $endDate, int $page, int $perPage = 25): array
    {
        $range = $this->normalizeRange($startDate, $endDate);
        $page = max(1, $page);
        $offset = ($page - 1) * $perPage;
        $result = $this->audits->search($range['startDateTime'], $range['endDateTime'], $perPage, $offset);
        $total = $result['total'];
        $pages = (int) max(1, $total > 0 ? ceil($total / $perPage) : 1);

        if ($page > $pages && $total > 0) {
            $page = $pages;
            $offset = ($page - 1) * $perPage;
            $result = $this->audits->search($range['startDateTime'], $range['endDateTime'], $perPage, $offset);
        }

        return [
            'items' => $result['rows'],
            'pagination' => [
                'page' => $page,
                'pages' => $pages,
                'per_page' => $perPage,
                'total' => $total,
            ],
        ];
    }

    /**
     * @return array<string, mixed>
     */
    private function buildFileReport(string $path, ?string $startDate, ?string $endDate, int $page, int $perPage): array
    {
        $range = $this->normalizeRange($startDate, $endDate);
        $entries = $this->collectLogEntries($path, $range['startTimestamp'], $range['endTimestamp']);
        $total = count($entries);
        $page = max(1, $page);
        $pages = (int) max(1, $total > 0 ? ceil($total / $perPage) : 1);

        if ($page > $pages && $total > 0) {
            $page = $pages;
        }

        $offset = ($page - 1) * $perPage;
        $items = array_slice($entries, $offset, $perPage);

        // Remove internal timestamp helper before returning.
        foreach ($items as &$item) {
            unset($item['timestamp_unix']);
        }
        unset($item);

        return [
            'items' => $items,
            'pagination' => [
                'page' => $page,
                'pages' => $pages,
                'per_page' => $perPage,
                'total' => $total,
            ],
        ];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    private function collectLogEntries(string $path, ?int $startTimestamp, ?int $endTimestamp): array
    {
        if (!is_file($path) || !is_readable($path)) {
            return [];
        }

        $lines = @file($path, FILE_IGNORE_NEW_LINES);
        if ($lines === false) {
            return [];
        }

        $entries = [];
        for ($index = count($lines) - 1; $index >= 0; $index--) {
            $entry = $this->parseLogLine((string) $lines[$index]);
            $timestamp = $entry['timestamp'] !== null ? strtotime($entry['timestamp']) : null;

            if (($startTimestamp !== null || $endTimestamp !== null) && $timestamp === null) {
                continue;
            }

            if ($startTimestamp !== null && $timestamp !== null && $timestamp < $startTimestamp) {
                // Since we iterate from newest to oldest, once we pass the lower bound we can stop.
                break;
            }

            if ($endTimestamp !== null && $timestamp !== null && $timestamp > $endTimestamp) {
                continue;
            }

            $entry['timestamp_unix'] = $timestamp;
            $entries[] = $entry;
        }

        return $entries;
    }

    /**
     * @return array<string, mixed>
     */
    private function normalizeRange(?string $startDate, ?string $endDate): array
    {
        $startTimestamp = null;
        $endTimestamp = null;
        $startDateTime = null;
        $endDateTime = null;

        if ($startDate !== null) {
            $startDateTime = $startDate . ' 00:00:00';
            $startTimestamp = strtotime($startDateTime) ?: null;
        }

        if ($endDate !== null) {
            $endDateTime = $endDate . ' 23:59:59';
            $endTimestamp = strtotime($endDateTime) ?: null;
        }

        return [
            'startTimestamp' => $startTimestamp,
            'endTimestamp' => $endTimestamp,
            'startDateTime' => $startDateTime,
            'endDateTime' => $endDateTime,
        ];
    }

    /**
     * @return array<string, string|null>
     */
    private function parseLogLine(string $line): array
    {
        $trimmed = trim($line);
        if ($trimmed === '') {
            return [
                'raw' => '',
                'timestamp' => null,
                'level' => null,
                'message' => '',
            ];
        }

        if (preg_match('/^\[(.*?)\]\s([A-Z]+)(?::)?\s(.*)$/', $trimmed, $matches) === 1) {
            return [
                'raw' => $trimmed,
                'timestamp' => $matches[1],
                'level' => $matches[2],
                'message' => $matches[3],
            ];
        }

        if (preg_match('/^\[(.*?)\]\s(.*)$/', $trimmed, $fallback) === 1) {
            return [
                'raw' => $trimmed,
                'timestamp' => $fallback[1],
                'level' => null,
                'message' => $fallback[2],
            ];
        }

        return [
            'raw' => $trimmed,
            'timestamp' => null,
            'level' => null,
            'message' => $trimmed,
        ];
    }
}
