<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\NotificationOutboxRepository;
use RuntimeException;

final class NotificationService
{
    private string $logFile;

    public function __construct(
        private readonly NotificationOutboxRepository $repository
    ) {
        $logDir = BASE_PATH . '/storage/logs';
        if (!is_dir($logDir)) {
            mkdir($logDir, 0775, true);
        }
        $this->logFile = $logDir . '/notifications.log';
    }

    /**
     * @param array<string, mixed> $ticket
     */
    public function queueTicketOpened(array $ticket): void
    {
        $recipient = $ticket['requester_email'] ?? null;
        if (!$this->isValidEmail($recipient)) {
            return;
        }

        $subject = sprintf('Chamado %s aberto', $ticket['reference'] ?? '');
        $body = $this->renderTemplate('Seu chamado foi registrado com sucesso.', $ticket);
        $this->enqueue('ticket_opened', $ticket['id'] ?? null, $recipient, $ticket['requester_name'] ?? null, $subject, $body);
    }

    /**
     * @param array<string, mixed> $ticket
     */
    public function queueTicketAssigned(array $ticket): void
    {
        $recipient = $ticket['assignee_email'] ?? null;
        if (!$this->isValidEmail($recipient)) {
            return;
        }

        $subject = sprintf('Novo chamado atribuído: %s', $ticket['reference'] ?? '');
        $body = $this->renderTemplate('Você foi designado para este chamado.', $ticket);
        $this->enqueue('ticket_assigned', $ticket['id'] ?? null, $recipient, $ticket['assignee_name'] ?? null, $subject, $body);
    }

    /**
     * @param array<string, mixed> $ticket
     */
    public function queueStatusUpdate(array $ticket, string $newStatus): void
    {
        $recipient = $ticket['requester_email'] ?? null;
        if (!$this->isValidEmail($recipient)) {
            return;
        }

        $subject = sprintf('Atualização do chamado %s', $ticket['reference'] ?? '');
        $body = $this->renderTemplate(
            sprintf('O status do seu chamado mudou para %s.', strtoupper($newStatus)),
            $ticket
        );
        $this->enqueue('ticket_status_' . $newStatus, $ticket['id'] ?? null, $recipient, $ticket['requester_name'] ?? null, $subject, $body);
    }

    public function processPending(int $batchSize = 20): void
    {
        $pending = $this->repository->pending($batchSize);
        foreach ($pending as $message) {
            try {
                $this->deliver($message);
                $this->repository->markSent((int) $message['id']);
            } catch (RuntimeException $exception) {
                $this->repository->markError((int) $message['id'], $exception->getMessage());
                $this->log('ERROR', sprintf('Falha ao enviar notificação #%d: %s', $message['id'], $exception->getMessage()));
            }
        }
    }

    /**
     * @param array<string, mixed> $message
     */
    private function deliver(array $message): void
    {
        $recipient = $message['recipient_email'] ?? '';
        $subject = $message['subject'] ?? '';
        $body = $message['body'] ?? '';

        if (!$this->isValidEmail($recipient) || trim($subject) === '') {
            throw new RuntimeException('Dados do destinatário inválidos.');
        }

        $emailBody = sprintf(
            "Para: %s\nAssunto: %s\nEvento: %s\n\n%s\n\n--------\n",
            $recipient,
            $subject,
            $message['event_type'] ?? 'event',
            $body
        );

        file_put_contents($this->logFile, $emailBody, FILE_APPEND);
    }

    private function enqueue(?string $event, $ticketId, string $recipientEmail, ?string $recipientName, string $subject, string $body): void
    {
        $this->repository->insert([
            'ticket_id' => $ticketId,
            'event_type' => (string) $event,
            'recipient_email' => $recipientEmail,
            'recipient_name' => $recipientName,
            'subject' => $subject,
            'body' => $body,
        ]);
    }

    /**
     * @param array<string, mixed> $ticket
     */
    private function renderTemplate(string $intro, array $ticket): string
    {
        return sprintf(
            "%s\n\nReferência: %s\nAssunto: %s\nStatus: %s\nFila: %s\n\n- HelpDesk Sinarco",
            $intro,
            $ticket['reference'] ?? '-',
            $ticket['subject'] ?? '-',
            $ticket['status_name'] ?? '-',
            $ticket['queue_name'] ?? '-'
        );
    }

    private function isValidEmail(?string $email): bool
    {
        return $email !== null && filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }

    private function log(string $level, string $message): void
    {
        $line = sprintf('[%s] %s %s%s', date('Y-m-d H:i:s'), strtoupper($level), $message, PHP_EOL);
        file_put_contents($this->logFile, $line, FILE_APPEND);
    }
}
