<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\ThemeRepository;

final class ThemeService
{
    private array $cache = [];

    private array $defaults = [
        'primary_color' => '#ff7a18',
        'secondary_color' => '#6c63ff',
        'dark_color' => '#1f1f1f',
        'accent_color' => '#37474f',
        'logo_text' => 'HelpDesk',
        'logo_path' => null,
    ];

    public function __construct(
        private readonly ThemeRepository $repository
    ) {
    }

    /**
     * @return array<string, string>
     */
    public function variables(string $context = 'global'): array
    {
        $theme = $this->theme($context);

        return [
            '--primary' => $theme['primary_color'] ?? $this->defaults['primary_color'],
            '--secondary' => $theme['secondary_color'] ?? $this->defaults['secondary_color'],
            '--dark' => $theme['dark_color'] ?? $this->defaults['dark_color'],
            '--accent' => $theme['accent_color'] ?? $this->defaults['accent_color'],
            '--bg' => '#f7f8fb',
            '--surface' => '#ffffff',
            '--text' => '#1f1f1f',
        ];
    }

    public function brand(string $context = 'global'): string
    {
        $theme = $this->theme($context);
        return $theme['logo_text'] ?? $this->defaults['logo_text'];
    }

    public function logo(string $context = 'global'): ?string
    {
        $theme = $this->theme($context);
        $path = $theme['logo_path'] ?? $this->defaults['logo_path'];
        return $path ?: null;
    }

    /**
     * @return array<string, string>
     */
    private function theme(string $context): array
    {
        if (!isset($this->cache[$context])) {
            $row = $context === 'global'
                ? ($this->repository->findByContext($context) ?? $this->repository->findDefault())
                : $this->repository->findByContext($context);

            $this->cache[$context] = $row ?? [];
        }

        return $this->cache[$context];
    }
}
