<?php
declare(strict_types=1);

namespace App\Services;

use App\Config\AppConfig;
use App\Repositories\CategoryRepository;
use App\Repositories\CommentRepository;
use App\Repositories\EmergencyActionRepository;
use App\Repositories\TicketFieldRepository;
use App\Repositories\TicketRepository;
use RuntimeException;

final class TicketService
{
    private bool $autoCloseExecuted = false;

    /**
     * @var array<string, int|null>
     */
    private array $statusIdCache = [];

    /**
     * @var array<int, string>
     */
    private array $statusSlugCache = [];

    /**
     * @var array<string, string[]>
     */
    private array $workflowTransitions = [
        'novo' => ['em-triagem', 'cancelado'],
        'em-triagem' => ['em-atendimento', 'cancelado'],
        'em-atendimento' => ['aguardando-cliente', 'resolvido', 'pausado-interno', 'cancelado'],
        'aguardando-cliente' => ['em-atendimento', 'resolvido', 'cancelado'],
        'pausado-interno' => ['em-atendimento', 'resolvido', 'cancelado'],
        'resolvido' => ['fechado', 'em-atendimento'],
        'fechado' => ['em-atendimento'],
    ];

    public function __construct(
        private readonly TicketRepository $tickets,
        private readonly CommentRepository $comments,
        private readonly AttachmentService $attachments,
        private readonly TicketFieldRepository $ticketFields,
        private readonly CategoryRepository $categories,
        private readonly EmergencyActionRepository $emergencyActions,
        private readonly SlaService $slaService,
        private readonly NotificationService $notificationService
    ) {
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function visibleTickets(array $user, array $filters = [], int $limit = 25): array
    {
        $this->autoCloseIfNeeded();
        $scopedFilters = array_merge($this->sanitizeFilters($filters), $this->filtersForUser($user));
        $tickets = $this->tickets->listByFilters($scopedFilters, $limit);
        foreach ($tickets as &$ticket) {
            $ticket['is_emergency'] = $this->isEmergencyTicket($ticket);
        }
        unset($ticket);

        return $tickets;
    }

    public function ticketForUser(int $ticketId, array $user): ?array
    {
        $this->autoCloseIfNeeded();
        $ticket = $this->tickets->findById($ticketId);
        if ($ticket === null) {
            return null;
        }

        $canView = \can('view_ticket', [
            'requester_id' => (int) $ticket['requester_id'],
            'queue_id' => (int) $ticket['queue_id'],
        ]);

        if (!$canView) {
            return null;
        }

        $ticket['is_emergency'] = $this->isEmergencyTicket($ticket);

        return $ticket;
    }

    /**
     * @param array<string, mixed> $input
     */
    public function createTicket(array $input, array $user): int
    {
        $this->autoCloseIfNeeded();
        $reference = $this->tickets->generateReference();
        $requesterId = $this->resolveRequesterId($input, $user);
        $categoryId = $this->nullableInt($input['category_id'] ?? null);
        $queueId = $this->resolveQueueId(
            $this->nullableInt($input['queue_id'] ?? null),
            $categoryId,
            $user
        );

        $data = [
            'reference' => $reference,
            'client_type' => $input['client_type'] ?? $this->inferClientType($user),
            'main_category' => $input['main_category'] ?? 'ti',
            'site_name' => $input['site_name'] ?? null,
            'unit_name' => $input['unit_name'] ?? null,
            'condominium' => $input['condominium'] ?? null,
            'customer_name' => $input['customer_name'] ?? null,
            'contact_name' => $input['contact_name'] ?? null,
            'contact_phone' => $input['contact_phone'] ?? null,
            'contact_email' => $input['contact_email'] ?? null,
            'subject' => trim((string) ($input['subject'] ?? '')),
            'description' => trim((string) ($input['description'] ?? '')),
            'requester_id' => $requesterId,
            'assignee_id' => $this->normalizeAssigneeId($this->nullableInt($input['assignee_id'] ?? null), $user),
            'queue_id' => $queueId,
            'category_id' => $categoryId,
            'priority_id' => $this->requiredInt($input['priority_id'] ?? null, 'Prioridade obrigatoria'),
            'status_id' => $this->requiredInt($input['status_id'] ?? null, 'Status obrigatorio'),
            'sla_due_at' => $input['sla_due_at'] ?? null,
            'customer_visible' => isset($input['customer_visible']) ? (int) (bool) $input['customer_visible'] : 1,
        ];

        if ($data['subject'] === '' || $data['description'] === '') {
            throw new RuntimeException('Assunto e descricao sao obrigatorios.');
        }
        $statusSlug = $this->statusSlug((int) $data['status_id']);
        $data = $this->applyStatusTimestampsOnCreate($data, $statusSlug);

        $ticketId = $this->tickets->create($data);
        $this->tickets->insertHistory($ticketId, null, $data['status_id'], $user['id'] ?? null, 'Ticket criado');
        $this->handleDynamicFields(
            $ticketId,
            isset($data['category_id']) ? (int) $data['category_id'] : null,
            $input['fields'] ?? []
        );
        $freshTicket = $this->tickets->findById($ticketId);
        if ($freshTicket !== null) {
            $this->notificationService->queueTicketOpened($freshTicket);
            if (!empty($freshTicket['assignee_id'])) {
                $this->notificationService->queueTicketAssigned($freshTicket);
            }
        }

        \app_log('INFO', 'Ticket criado', [
            'ticket_id' => $ticketId,
            'reference' => $reference,
            'user_id' => $user['id'] ?? null,
            'status_id' => (int) $data['status_id'],
            'queue_id' => (int) $data['queue_id'],
            'assignee_id' => $data['assignee_id'] ?? null,
        ]);

        \audit_log('ticket', 'create', $ticketId, [
            'reference' => $reference,
            'status_id' => (int) $data['status_id'],
            'queue_id' => (int) $data['queue_id'],
            'assignee_id' => $data['assignee_id'] ?? null,
            'client_type' => $data['client_type'],
        ]);

        return $ticketId;
    }

    /**
     * @param array<string, mixed> $input
     */
    public function updateTicket(int $ticketId, array $input, array $user): void
    {
        $this->autoCloseIfNeeded();
        $existing = $this->ticketForUser($ticketId, $user);
        if ($existing === null) {
            throw new RuntimeException('Ticket nao encontrado ou sem permissao.');
        }

        if (!\can('update_ticket')) {
            throw new RuntimeException('Usuario sem permissao para atualizar o ticket.');
        }

        $payload = [
            'client_type' => $input['client_type'] ?? $existing['client_type'],
            'main_category' => $input['main_category'] ?? $existing['main_category'],
            'site_name' => $input['site_name'] ?? $existing['site_name'],
            'unit_name' => $input['unit_name'] ?? $existing['unit_name'],
            'condominium' => $input['condominium'] ?? $existing['condominium'],
            'customer_name' => $input['customer_name'] ?? $existing['customer_name'],
            'contact_name' => $input['contact_name'] ?? $existing['contact_name'],
            'contact_phone' => $input['contact_phone'] ?? $existing['contact_phone'],
            'contact_email' => $input['contact_email'] ?? $existing['contact_email'],
            'subject' => trim((string) ($input['subject'] ?? $existing['subject'])),
            'description' => trim((string) ($input['description'] ?? $existing['description'])),
            'assignee_id' => $this->normalizeAssigneeId(
                $this->nullableInt($input['assignee_id'] ?? $existing['assignee_id']),
                $user,
                isset($existing['assignee_id']) ? (int) $existing['assignee_id'] : null
            ),
            'category_id' => $this->nullableInt($input['category_id'] ?? $existing['category_id']),
            'priority_id' => $this->requiredInt($input['priority_id'] ?? $existing['priority_id'], 'Prioridade obrigatoria'),
            'status_id' => $this->requiredInt($input['status_id'] ?? $existing['status_id'], 'Status obrigatorio'),
            'customer_visible' => isset($input['customer_visible'])
                ? (int) (bool) $input['customer_visible']
                : 1,
        ];
        $payload['queue_id'] = $this->resolveQueueId(
            $this->nullableInt($input['queue_id'] ?? null),
            $payload['category_id'],
            $user,
            (int) $existing['queue_id']
        );

        $currentSlug = (string) ($existing['status_slug'] ?? $this->statusSlug((int) $existing['status_id']));
        $newSlug = $this->statusSlug((int) $payload['status_id']);
        $assignmentChanged = ((int) ($existing['assignee_id'] ?? 0)) !== (int) ($payload['assignee_id'] ?? 0);

        $this->validateStatusTransition($currentSlug, $newSlug, $existing);
        $payload = $this->applyStatusTimestampsOnUpdate($payload, $currentSlug, $newSlug);

        $this->tickets->update($ticketId, $payload);
        $this->handleDynamicFields(
            $ticketId,
            isset($payload['category_id']) ? (int) $payload['category_id'] : null,
            $input['fields'] ?? []
        );

        $changes = $this->detectAuditChanges($existing, $payload);
        if ($changes !== []) {
            \app_log('INFO', 'Ticket atualizado', [
                'ticket_id' => $ticketId,
                'user_id' => $user['id'] ?? null,
                'changes' => $changes,
            ]);

            \audit_log('ticket', 'update', $ticketId, [
                'changes' => $changes,
                'from_status' => $currentSlug,
                'to_status' => $newSlug,
            ]);
        }

        if ((int) $existing['status_id'] !== $payload['status_id']) {
            $this->tickets->insertHistory(
                $ticketId,
                (int) $existing['status_id'],
                $payload['status_id'],
                $user['id'] ?? null,
                $input['status_notes'] ?? 'Status atualizado'
            );
        }

        $updatedTicket = $this->tickets->findById($ticketId);
        if ($updatedTicket !== null) {
            if ($assignmentChanged && !empty($updatedTicket['assignee_id'])) {
                $this->notificationService->queueTicketAssigned($updatedTicket);
            }

            if (in_array($newSlug, ['em-atendimento', 'resolvido'], true)) {
                $this->notificationService->queueStatusUpdate($updatedTicket, $newSlug);
            }
        }
    }

    public function deleteTicket(int $ticketId, array $user): void
    {
        if (!\can('delete_ticket')) {
            throw new RuntimeException('Usuario sem permissao para excluir tickets.');
        }

        $ticket = $this->tickets->findById($ticketId);
        if ($ticket === null) {
            throw new RuntimeException('Ticket nao encontrado.');
        }

        $this->tickets->delete($ticketId);

        \app_log('WARNING', 'Ticket excluido', [
            'ticket_id' => $ticketId,
            'reference' => $ticket['reference'] ?? null,
            'user_id' => $user['id'] ?? null,
        ]);

        \audit_log('ticket', 'delete', $ticketId, [
            'reference' => $ticket['reference'] ?? null,
        ]);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function timeline(int $ticketId, array $user): array
    {
        $ticket = $this->ticketForUser($ticketId, $user);
        if ($ticket === null) {
            throw new RuntimeException('Ticket nao encontrado ou sem permissao.');
        }

        $statusHistory = $this->tickets->statusHistory($ticketId);
        $comments = $this->filterCommentsForUser($this->comments->listByTicket($ticketId), $user);
        $commentIds = array_map(static fn (array $comment): int => (int) $comment['id'], $comments);
        $attachments = $this->attachments->attachmentsGroupedByComment($commentIds);
        $emergencyLogs = $this->isEmergencyTicket($ticket)
            ? $this->emergencyActions->listByTicket($ticketId)
            : [];
        $events = [];

        foreach ($statusHistory as $event) {
            $events[] = [
                'type' => 'status',
                'timestamp' => $event['changed_at'] ?? $event['created_at'] ?? '',
                'from_status' => $event['from_status_name'] ?? null,
                'to_status' => $event['to_status_name'] ?? null,
                'notes' => $event['notes'] ?? null,
                'author' => $event['changed_by_name'] ?? 'Sistema',
            ];
        }

        foreach ($comments as $comment) {
            $commentId = (int) $comment['id'];
            $events[] = [
                'type' => 'comment',
                'timestamp' => $comment['created_at'],
                'author' => $comment['author_name'] ?? 'Usuário',
                'content' => $comment['content'],
                'is_internal' => (bool) $comment['is_internal'],
                'attachments' => $attachments[$commentId] ?? [],
            ];
        }

        foreach ($emergencyLogs as $log) {
            $events[] = [
                'type' => 'emergency',
                'timestamp' => $log['occurred_at'] ?? $log['created_at'] ?? '',
                'action_type' => $log['action_type'] ?? '',
                'authority' => $log['authority'] ?? '',
                'contact_name' => $log['contact_name'] ?? '',
                'contact_phone' => $log['contact_phone'] ?? '',
                'protocol_code' => $log['protocol_code'] ?? '',
                'notes' => $log['notes'] ?? '',
                'author' => $log['created_by_name'] ?? 'NOC',
            ];
        }

        usort($events, static function (array $a, array $b): int {
            return strcmp((string) ($b['timestamp'] ?? ''), (string) ($a['timestamp'] ?? ''));
        });

        return $events;
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function emergencyActionLogs(int $ticketId, array $user): array
    {
        $ticket = $this->ticketForUser($ticketId, $user);
        if ($ticket === null || !$this->isEmergencyTicket($ticket)) {
            return [];
        }

        return $this->emergencyActions->listByTicket($ticketId);
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed> $files
     */
    public function addComment(int $ticketId, array $input, array $files, array $user): void
    {
        $ticket = $this->ticketForUser($ticketId, $user);
        if ($ticket === null) {
            throw new RuntimeException('Ticket nao encontrado ou sem permissao.');
        }

        $content = trim((string) ($input['content'] ?? ''));
        $fileInput = $files['attachments'] ?? null;
        $hasUploads = is_array($fileInput) && $this->attachments->hasUploads($fileInput);

        if ($content === '' && !$hasUploads) {
            throw new RuntimeException('Informe um comentário ou envie ao menos um anexo.');
        }

        $authorId = isset($user['id']) ? (int) $user['id'] : null;
        $isInternal = isset($input['is_internal']) && \can('update_ticket');
        $commentId = $this->comments->create($ticketId, $authorId, $content, $isInternal);

        $savedAttachments = [];
        if ($hasUploads) {
            $savedAttachments = $this->attachments->storeTicketFiles($ticketId, $commentId, $fileInput);
        }

        \app_log('INFO', 'Comentario adicionado ao ticket', [
            'ticket_id' => $ticketId,
            'comment_id' => $commentId,
            'user_id' => $authorId,
            'is_internal' => $isInternal,
            'attachments' => count($savedAttachments),
        ]);

        \audit_log('ticket_comment', 'create', $commentId, [
            'ticket_id' => $ticketId,
            'is_internal' => $isInternal,
            'has_attachments' => $savedAttachments !== [],
        ]);
    }

    public function attachmentForUser(int $attachmentId, array $user): ?array
    {
        $attachment = $this->attachments->find($attachmentId);
        if ($attachment === null) {
            return null;
        }

        $ticket = $this->ticketForUser((int) $attachment['ticket_id'], $user);
        if ($ticket === null) {
            return null;
        }

        if (($attachment['comment_is_internal'] ?? null) && !\can('update_ticket')) {
            return null;
        }

        return $attachment;
    }

    /**
     * @return array<int, array<int, array<string, mixed>>>
     */
    public function fieldCatalog(): array
    {
        return $this->ticketFields->fieldsGroupedByCategory();
    }

    /**
     * @return array<string, string>
     */
    public function ticketFieldValues(int $ticketId): array
    {
        return $this->ticketFields->valueMapForTicket($ticketId);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function ticketFieldDisplay(int $ticketId): array
    {
        return $this->ticketFields->displayFieldsForTicket($ticketId);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function queueSummary(array $user): array
    {
        $queueId = (int) ($user['queue_id'] ?? 0);
        if ($queueId === 0) {
            return [];
        }

        return $this->tickets->queueStatusSummary($queueId);
    }

    /**
     * @return array<string, mixed>
     */
    public function slaMetrics(array $ticket): array
    {
        if (!isset($ticket['id'])) {
            return ['policy' => ['response_minutes' => 0, 'resolution_minutes' => 0, 'source' => 'default']];
        }

        $history = $this->tickets->statusHistory((int) $ticket['id']);
        return $this->slaService->metrics($ticket, $history);
    }

    public function logEmergencyAction(int $ticketId, array $input, array $user): void
    {
        $ticket = $this->ticketForUser($ticketId, $user);
        if ($ticket === null || !$this->isEmergencyTicket($ticket)) {
            throw new RuntimeException('Chamado inválido para registro de emergência.');
        }

        if (!\can('update_ticket')) {
            throw new RuntimeException('Você não possui permissão para registrar acionamentos.');
        }

        $actionType = $this->sanitizeActionType((string) ($input['action_type'] ?? ''));
        $authority = trim((string) ($input['authority'] ?? ''));
        $contactName = trim((string) ($input['contact_name'] ?? ''));
        $contactPhone = trim((string) ($input['contact_phone'] ?? ''));
        $protocol = trim((string) ($input['protocol_code'] ?? ''));
        $notes = trim((string) ($input['notes'] ?? ''));
        $occurredAt = trim((string) ($input['occurred_at'] ?? ''));

        if ($authority === '') {
            throw new RuntimeException('Informe qual autoridade foi acionada.');
        }

        if ($occurredAt === '') {
            throw new RuntimeException('Informe o horário do acionamento.');
        }

        $parsedDate = strtotime($occurredAt);
        if ($parsedDate === false) {
            throw new RuntimeException('Horário de acionamento inválido.');
        }

        $actionId = $this->emergencyActions->create([
            'ticket_id' => $ticketId,
            'action_type' => $actionType,
            'authority' => $authority,
            'contact_name' => $contactName !== '' ? $contactName : null,
            'contact_phone' => $contactPhone !== '' ? $contactPhone : null,
            'protocol_code' => $protocol !== '' ? $protocol : null,
            'occurred_at' => date('Y-m-d H:i:s', $parsedDate),
            'notes' => $notes !== '' ? $notes : null,
            'created_by' => $user['id'] ?? null,
        ]);

        \app_log('INFO', 'Registro de emergencia criado', [
            'ticket_id' => $ticketId,
            'action_id' => $actionId,
            'user_id' => $user['id'] ?? null,
            'action_type' => $actionType,
        ]);

        \audit_log('ticket_emergency', 'create', $actionId, [
            'ticket_id' => $ticketId,
            'action_type' => $actionType,
            'authority' => $authority,
        ]);
    }

    /**
     * @return array<string, int>
     */
    private function filtersForUser(array $user): array
    {
        $role = $user['role_slug'] ?? '';

        if ($role === 'gestor') {
            return [];
        }

        if (in_array($role, ['atendente-ti', 'atendente-noc'], true)) {
            return $this->queueFilter($user);
        }

        return ['requester_id' => (int) $user['id']];
    }

    /**
     * @return array<string, int>
     */
    private function queueFilter(array $user): array
    {
        $queueId = (int) ($user['queue_id'] ?? 0);
        if ($queueId === 0) {
            throw new RuntimeException('Usuario atendente sem fila vinculada.');
        }

        return ['queue_id' => $queueId];
    }

    /**
     * @param array<string, mixed> $filters
     * @return array<string, mixed>
     */
    private function sanitizeFilters(array $filters): array
    {
        $allowed = [
            'status_id',
            'priority_id',
            'category_id',
            'queue_id',
            'main_category',
            'from_date',
            'to_date',
            'assignee_id',
            'unassigned',
        ];
        $clean = [];

        foreach ($allowed as $key) {
            if (!isset($filters[$key]) || $filters[$key] === '' || $filters[$key] === null) {
                continue;
            }

            if (in_array($key, ['status_id', 'priority_id', 'category_id', 'queue_id', 'assignee_id'], true)) {
                $clean[$key] = (int) $filters[$key];
            } elseif (in_array($key, ['from_date', 'to_date'], true)) {
                $clean[$key] = $filters[$key];
            } elseif ($key === 'unassigned') {
                $clean[$key] = (int) (bool) $filters[$key];
            } else {
                $clean[$key] = $filters[$key];
            }
        }

        return $clean;
    }

    /**
     * @param array<string, mixed> $input
     */
    private function resolveRequesterId(array $input, array $user): int
    {
        $requesterId = (int) ($input['requester_id'] ?? 0);
        if ($requesterId > 0 && \can('view_all_tickets')) {
            return $requesterId;
        }

        return (int) $user['id'];
    }

    private function inferClientType(array $user): string
    {
        return $user['role_slug'] === 'externo' ? 'externo' : 'interno';
    }

    public function isEmergencyTicket(array $ticket): bool
    {
        $categorySlug = strtolower((string) ($ticket['category_slug'] ?? ''));
        if ($categorySlug !== '') {
            if (str_contains($categorySlug, 'emergencia') || str_contains($categorySlug, 'emergência')) {
                return true;
            }
        }

        $mainCategory = strtolower((string) ($ticket['main_category'] ?? ''));
        if ($mainCategory !== 'noc') {
            return false;
        }

        $priorityName = strtolower((string) ($ticket['priority_name'] ?? ''));
        $priorityId = (int) ($ticket['priority_id'] ?? 0);

        return $priorityId >= 4 || str_contains($priorityName, 'crit');
    }

    private function normalizeAssigneeId(?int $assigneeId, array $user, ?int $currentAssignee = null): ?int
    {
        $role = $user['role_slug'] ?? '';
        if (in_array($role, ['interno', 'externo'], true)) {
            return $currentAssignee;
        }

        if ($this->isGestor($user)) {
            if ($assigneeId === null || $assigneeId === 0) {
                return null;
            }

            return $assigneeId;
        }

        if (in_array($role, ['atendente-ti', 'atendente-noc'], true)) {
            if ($assigneeId === null || $assigneeId === 0) {
                return null;
            }

            if ($assigneeId !== (int) $user['id']) {
                throw new RuntimeException('Somente gestores podem reatribuir responsáveis para outros usuários.');
            }

            return $assigneeId;
        }

        return $assigneeId;
    }

    private function nullableInt(mixed $value): ?int
    {
        if ($value === null || $value === '') {
            return null;
        }

        return (int) $value;
    }

    private function requiredInt(mixed $value, string $message): int
    {
        if ($value === null || $value === '' || (int) $value === 0) {
            throw new RuntimeException($message);
        }

        return (int) $value;
    }

    private function sanitizeActionType(string $value): string
    {
        $value = strtolower(trim($value));
        $allowed = [
            'alerta',
            'contato',
            'acionamento',
            'conclusao',
        ];

        if (in_array($value, $allowed, true)) {
            return $value;
        }

        return 'acionamento';
    }

    /**
     * @param array<int, array<string, mixed>> $comments
     * @return array<int, array<string, mixed>>
     */
    private function filterCommentsForUser(array $comments, array $user): array
    {
        $role = $user['role_slug'] ?? '';
        $canSeeInternal = in_array($role, ['gestor', 'atendente-ti', 'atendente-noc'], true);

        if ($canSeeInternal) {
            return $comments;
        }

        return array_values(array_filter($comments, static function (array $comment): bool {
            return (int) ($comment['is_internal'] ?? 0) === 0;
        }));
    }

    /**
     * @param array<string, mixed>|null $fieldInput
     */
    private function handleDynamicFields(int $ticketId, ?int $categoryId, mixed $fieldInput): void
    {
        if (!is_array($fieldInput)) {
            $fieldInput = [];
        }

        $this->ticketFields->deleteValuesForTicket($ticketId);

        if ($categoryId === null || $categoryId === 0) {
            return;
        }

        $definitions = $this->ticketFields->fieldsByCategory($categoryId);
        if ($definitions === []) {
            return;
        }

        $valuesToPersist = [];

        foreach ($definitions as $definition) {
            $name = $definition['name'];
            $hasInput = array_key_exists($name, $fieldInput);
            $raw = $fieldInput[$name] ?? null;
            $value = $this->normalizeFieldValue($definition, $raw, $hasInput);

            if ($definition['type'] === 'boolean') {
                $valuesToPersist[(int) $definition['id']] = $value === '1' ? '1' : '0';
                continue;
            }

            if ($definition['type'] === 'select' && $value !== null && $value !== '') {
                $options = $definition['options'] ?? [];
                if (!in_array($value, $options, true)) {
                    throw new RuntimeException('Valor inválido para "' . $definition['label'] . '".');
                }
            }

            if ($definition['type'] === 'number' && $value !== null && $value !== '') {
                if (!is_numeric($value)) {
                    throw new RuntimeException('O campo "' . $definition['label'] . '" deve ser numérico.');
                }
            }

            if (($value === null || $value === '') && $definition['is_required']) {
                throw new RuntimeException('Campo "' . $definition['label'] . '" é obrigatório.');
            }

            if ($value !== null && $value !== '') {
                $valuesToPersist[(int) $definition['id']] = $value;
            }
        }

        if ($valuesToPersist !== []) {
            $this->ticketFields->saveValues($ticketId, $valuesToPersist);
        }
    }

    private function normalizeFieldValue(array $field, mixed $value, bool $hasInput): ?string
    {
        $type = $field['type'];

        if ($type === 'boolean') {
            if ($hasInput && ((string) $value === '1' || $value === true || $value === 1)) {
                return '1';
            }

            return '0';
        }

        if (is_array($value)) {
            throw new RuntimeException('Valor inválido informado para "' . $field['label'] . '".');
        }

        $stringValue = $value === null ? null : trim((string) $value);

        if ($type === 'number' && $stringValue !== null) {
            return $stringValue;
        }

        return $stringValue;
    }

    private function autoCloseIfNeeded(): void
    {
        if ($this->autoCloseExecuted) {
            return;
        }

        $this->autoCloseExecuted = true;
        $this->autoCloseExpiredTickets();
    }

    private function autoCloseExpiredTickets(): void
    {
        $hours = AppConfig::autoCloseResolvedHours();
        if ($hours <= 0) {
            return;
        }

        $resolvedStatusId = $this->statusIdBySlug('resolvido');
        $closedStatusId = $this->statusIdBySlug('fechado');

        if ($resolvedStatusId === null || $closedStatusId === null) {
            return;
        }

        $limitDate = date('Y-m-d H:i:s', strtotime('-' . $hours . ' hours'));
        $tickets = $this->tickets->ticketsToAutoClose($resolvedStatusId, $limitDate);
        if ($tickets === []) {
            return;
        }

        foreach ($tickets as $ticket) {
            $ticketId = (int) $ticket['id'];
            $this->tickets->update($ticketId, [
                'status_id' => $closedStatusId,
                'closed_at' => date('Y-m-d H:i:s'),
            ]);

            $this->tickets->insertHistory(
                $ticketId,
                $resolvedStatusId,
                $closedStatusId,
                null,
                'Encerrado automaticamente após ' . $hours . 'h em resolvido.'
            );

            \app_log('INFO', 'Ticket autoencerrado por inatividade', [
                'ticket_id' => $ticketId,
                'resolved_at' => $ticket['resolved_at'] ?? null,
                'limit_hours' => $hours,
            ]);

            \audit_log('ticket', 'auto_close', $ticketId, [
                'resolved_at' => $ticket['resolved_at'] ?? null,
                'limit_hours' => $hours,
                'system' => true,
            ]);
        }
    }

    private function statusSlug(int $statusId): string
    {
        if (isset($this->statusSlugCache[$statusId])) {
            return $this->statusSlugCache[$statusId];
        }

        $slug = $this->tickets->statusSlugById($statusId);
        if ($slug === null) {
            throw new RuntimeException('Status desconhecido.');
        }

        $this->statusSlugCache[$statusId] = $slug;

        return $slug;
    }

    private function statusIdBySlug(string $slug): ?int
    {
        if (array_key_exists($slug, $this->statusIdCache)) {
            return $this->statusIdCache[$slug];
        }

        $id = $this->tickets->statusIdBySlug($slug);
        $this->statusIdCache[$slug] = $id;

        return $id;
    }

    private function validateStatusTransition(string $currentSlug, string $newSlug, array $existing): void
    {
        if ($currentSlug === $newSlug) {
            return;
        }

        if (!array_key_exists($currentSlug, $this->workflowTransitions)) {
            return;
        }

        $allowed = $this->workflowTransitions[$currentSlug];
        if (!in_array($newSlug, $allowed, true)) {
            throw new RuntimeException(sprintf(
                'Transição de %s para %s não é permitida.',
                strtoupper($currentSlug),
                strtoupper($newSlug)
            ));
        }

        if (in_array($currentSlug, ['resolvido', 'fechado'], true) && $newSlug !== 'fechado') {
            $timestamp = $currentSlug === 'fechado'
                ? ($existing['closed_at'] ?? null)
                : ($existing['resolved_at'] ?? null);

            if (!$this->withinReopenWindow($timestamp)) {
                throw new RuntimeException('Janela de reabertura expirou para este chamado.');
            }
        }
    }

    private function withinReopenWindow(?string $timestamp): bool
    {
        if ($timestamp === null || $timestamp === '') {
            return false;
        }

        $windowDays = AppConfig::ticketReopenDays();
        if ($windowDays <= 0) {
            return false;
        }

        $limit = strtotime('-' . $windowDays . ' days');
        $time = strtotime($timestamp);

        return $time !== false && $time >= $limit;
    }

    /**
     * @param array<string, mixed> $data
     * @return array<string, mixed>
     */
    private function applyStatusTimestampsOnCreate(array $data, string $newSlug): array
    {
        if ($newSlug === 'resolvido') {
            $data['resolved_at'] = date('Y-m-d H:i:s');
            $data['closed_at'] = null;
        } elseif ($newSlug === 'fechado') {
            $now = date('Y-m-d H:i:s');
            $data['resolved_at'] = $now;
            $data['closed_at'] = $now;
        }

        return $data;
    }

    /**
     * @param array<string, mixed> $payload
     * @return array<string, mixed>
     */
    private function applyStatusTimestampsOnUpdate(array $payload, string $currentSlug, string $newSlug): array
    {
        if ($currentSlug !== $newSlug) {
            if ($currentSlug === 'resolvido' && $newSlug !== 'resolvido') {
                $payload['resolved_at'] = null;
            }

            if ($currentSlug === 'fechado' && $newSlug !== 'fechado') {
                $payload['closed_at'] = null;
            }
        }

        if ($newSlug === 'resolvido') {
            $payload['resolved_at'] = date('Y-m-d H:i:s');
            $payload['closed_at'] = null;
        }

        if ($newSlug === 'fechado') {
            $payload['closed_at'] = date('Y-m-d H:i:s');
            if (!isset($payload['resolved_at']) || $payload['resolved_at'] === null) {
                $payload['resolved_at'] = date('Y-m-d H:i:s');
            }
        }

        return $payload;
    }

    private function resolveQueueId(?int $selectedQueueId, ?int $categoryId, array $user, ?int $currentQueueId = null): int
    {
        $categoryQueueId = $categoryId ? $this->categories->queueIdForCategory((int) $categoryId) : null;
        $queueId = null;

        if ($this->isGestor($user)) {
            $queueId = $selectedQueueId ?: $categoryQueueId ?: $currentQueueId;
        } else {
            if ($categoryQueueId !== null) {
                $queueId = $categoryQueueId;
            } elseif ($currentQueueId !== null) {
                $queueId = $currentQueueId;
            } else {
                $queueId = $selectedQueueId;
            }
        }

        if ($queueId === null || $queueId === 0) {
            throw new RuntimeException('Fila obrigatoria para o chamado.');
        }

        $this->assertQueueAccess($queueId, $user);

        return (int) $queueId;
    }

    private function assertQueueAccess(int $queueId, array $user): void
    {
        $role = $user['role_slug'] ?? '';
        if (!in_array($role, ['atendente-ti', 'atendente-noc'], true)) {
            return;
        }

        $userQueue = (int) ($user['queue_id'] ?? 0);
        if ($userQueue === 0) {
            throw new RuntimeException('Usuário atendente sem fila vinculada.');
        }

        if ($queueId !== $userQueue) {
            throw new RuntimeException('Você não pode mover o chamado para outra fila.');
        }
    }

    private function isGestor(array $user): bool
    {
        return ($user['role_slug'] ?? '') === 'gestor';
    }

    /**
     * @param array<string, mixed> $existing
     * @param array<string, mixed> $payload
     * @return array<string, array<string, mixed>>
     */
    private function detectAuditChanges(array $existing, array $payload): array
    {
        $keys = [
            'status_id',
            'assignee_id',
            'queue_id',
            'priority_id',
            'category_id',
            'customer_visible',
        ];

        $changes = [];

        foreach ($keys as $key) {
            $old = $this->normalizeAuditValue($existing[$key] ?? null);
            $new = $this->normalizeAuditValue($payload[$key] ?? ($existing[$key] ?? null));

            if ($old !== $new) {
                $changes[$key] = ['from' => $old, 'to' => $new];
            }
        }

        return $changes;
    }

    private function normalizeAuditValue(mixed $value): mixed
    {
        if ($value === null) {
            return null;
        }

        if (is_string($value) && ctype_digit($value)) {
            return (int) $value;
        }

        if (is_int($value)) {
            return $value;
        }

        if (is_bool($value)) {
            return $value ? 1 : 0;
        }

        return $value;
    }
}
