<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\AdminAuditRepository;
use App\Repositories\UserRepository;
use RuntimeException;

final class UserService
{
    public function __construct(
        private readonly UserRepository $users,
        private readonly AdminAuditRepository $audit
    ) {
    }

    /**
     * @return array{data: array<int, array<string, mixed>>, pagination: array<string, int>, filters: array<string, string|null>}
     */
    public function getPaginated(?string $search, ?string $status, int $page, int $perPage = 10): array
    {
        $search = $search !== null ? trim($search) : null;
        $status = in_array($status, ['active', 'inactive'], true) ? $status : null;
        $page = max(1, $page);

        $offset = ($page - 1) * $perPage;
        $result = $this->users->paginate($search, $status, $perPage, $offset);

        $pages = (int) max(1, ceil($result['total'] / $perPage));
        if ($page > $pages && $result['total'] > 0) {
            $page = $pages;
            $offset = ($page - 1) * $perPage;
            $result = $this->users->paginate($search, $status, $perPage, $offset);
        }

        return [
            'data' => $result['rows'],
            'pagination' => [
                'page' => $page,
                'pages' => $pages,
                'per_page' => $perPage,
                'total' => $result['total'],
            ],
            'filters' => [
                'search' => $search,
                'status' => $status,
            ],
        ];
    }

    public function find(int $id): ?array
    {
        return $this->users->findById($id);
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed> $actor
     */
    public function create(array $input, array $actor): int
    {
        $data = $this->validatePayload($input, requirePassword: true);
        $userId = $this->users->create($data);

        $this->audit->log(
            $actor['id'] ?? null,
            'user',
            'create',
            $userId,
            [
                'email' => $data['email'],
                'role_id' => $data['role_id'],
                'is_active' => $data['is_active'],
            ]
        );

        return $userId;
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed> $actor
     */
    public function update(int $id, array $input, array $actor): void
    {
        $user = $this->users->findById($id);
        if ($user === null) {
            throw new RuntimeException('Usuário não encontrado.');
        }

        $data = $this->validatePayload($input, requirePassword: false, ignoreId: $id);
        if (!$this->users->update($id, $data)) {
            throw new RuntimeException('Não foi possível atualizar o usuário.');
        }

        $this->audit->log(
            $actor['id'] ?? null,
            'user',
            'update',
            $id,
            [
                'email' => $data['email'],
                'role_id' => $data['role_id'],
                'is_active' => $data['is_active'],
            ]
        );
    }

    public function delete(int $id, array $actor): void
    {
        if (!$this->users->softDelete($id)) {
            throw new RuntimeException('Usuário não encontrado ou já removido.');
        }

        $this->audit->log($actor['id'] ?? null, 'user', 'delete', $id);
    }

    public function toggleStatus(int $id, bool $activate, array $actor): void
    {
        if (!$this->users->updateStatus($id, $activate ? 1 : 0)) {
            throw new RuntimeException('Não foi possível alterar o status do usuário.');
        }

        $this->audit->log($actor['id'] ?? null, 'user', 'toggle_status', $id, [
            'is_active' => $activate ? 1 : 0,
        ]);
    }

    /**
     * @param array<string, mixed> $input
     * @return array<string, mixed>
     */
    private function validatePayload(array $input, bool $requirePassword, ?int $ignoreId = null): array
    {
        $name = trim((string) ($input['name'] ?? ''));
        if (mb_strlen($name) < 3) {
            throw new RuntimeException('Nome deve ter pelo menos 3 caracteres.');
        }

        $email = strtolower(trim((string) ($input['email'] ?? '')));
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new RuntimeException('Informe um e-mail válido.');
        }

        if ($this->users->emailExists($email, $ignoreId)) {
            throw new RuntimeException('Já existe um usuário com este e-mail.');
        }

        $roleId = $this->requiredInt($input['role_id'] ?? null, 'Selecione o perfil do usuário.');
        $queueId = $this->nullableInt($input['queue_id'] ?? null);

        $password = (string) ($input['password'] ?? '');
        $passwordConfirmation = (string) ($input['password_confirmation'] ?? $password);

        if ($requirePassword || $password !== '') {
            if (strlen($password) < 6) {
                throw new RuntimeException('Senha deve ter pelo menos 6 caracteres.');
            }

            if ($password !== $passwordConfirmation) {
                throw new RuntimeException('Confirmação de senha não confere.');
            }
        }

        $payload = [
            'role_id' => $roleId,
            'queue_id' => $queueId,
            'name' => $name,
            'email' => $email,
            'phone' => trim((string) ($input['phone'] ?? '')) ?: null,
            'is_active' => isset($input['is_active']) ? 1 : 0,
        ];

        if ($requirePassword || $password !== '') {
            $payload['password_hash'] = password_hash($password, PASSWORD_BCRYPT);
        }

        return $payload;
    }

    private function nullableInt(mixed $value): ?int
    {
        if ($value === null || $value === '' || (int) $value === 0) {
            return null;
        }

        return (int) $value;
    }

    private function requiredInt(mixed $value, string $message): int
    {
        $intValue = $this->nullableInt($value);
        if ($intValue === null) {
            throw new RuntimeException($message);
        }

        return $intValue;
    }
}
