-- HelpDesk schema
SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci;
SET FOREIGN_KEY_CHECKS = 0;

CREATE DATABASE IF NOT EXISTS `helpdesk_sinarco`
  CHARACTER SET utf8mb4
  COLLATE utf8mb4_unicode_ci;
USE `helpdesk_sinarco`;

-- Drop tables to allow reinstall
DROP TABLE IF EXISTS admin_audit_logs;
DROP TABLE IF EXISTS notification_outbox;
DROP TABLE IF EXISTS ticket_field_values;
DROP TABLE IF EXISTS ticket_fields;
DROP TABLE IF EXISTS ticket_attachments;
DROP TABLE IF EXISTS ticket_emergency_actions;
DROP TABLE IF EXISTS ticket_comments;
DROP TABLE IF EXISTS ticket_status_history;
DROP TABLE IF EXISTS tickets;
DROP TABLE IF EXISTS categories;
DROP TABLE IF EXISTS statuses;
DROP TABLE IF EXISTS priorities;
DROP TABLE IF EXISTS sla_policies;
DROP TABLE IF EXISTS users;
DROP TABLE IF EXISTS queues;
DROP TABLE IF EXISTS roles;
DROP TABLE IF EXISTS settings_theme;

SET FOREIGN_KEY_CHECKS = 1;

CREATE TABLE roles (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(50) NOT NULL,
    name VARCHAR(100) NOT NULL,
    description VARCHAR(255) DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_roles_slug (slug)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE settings_theme (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    context VARCHAR(60) NOT NULL DEFAULT 'global',
    primary_color VARCHAR(7) NOT NULL DEFAULT '#ff6f00',
    secondary_color VARCHAR(7) NOT NULL DEFAULT '#6c63ff',
    dark_color VARCHAR(7) NOT NULL DEFAULT '#1f1f1f',
    accent_color VARCHAR(7) NOT NULL DEFAULT '#37474f',
    logo_text VARCHAR(120) DEFAULT 'HelpDesk',
    logo_path VARCHAR(255) DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_settings_theme_context (context)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE queues (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(80) NOT NULL,
    name VARCHAR(120) NOT NULL,
    description VARCHAR(255) DEFAULT NULL,
    sla_response_minutes SMALLINT UNSIGNED DEFAULT 0,
    sla_resolution_minutes SMALLINT UNSIGNED DEFAULT 0,
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_queues_slug (slug),
    UNIQUE KEY uq_queues_name (name),
    KEY idx_queues_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE users (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    role_id INT UNSIGNED NOT NULL,
    queue_id INT UNSIGNED DEFAULT NULL,
    name VARCHAR(150) NOT NULL,
    email VARCHAR(150) NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    phone VARCHAR(40) DEFAULT NULL,
    avatar VARCHAR(255) DEFAULT NULL,
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    last_login_at DATETIME DEFAULT NULL,
    deleted_at DATETIME DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_users_email (email),
    KEY idx_users_role (role_id),
    KEY idx_users_queue (queue_id),
    KEY idx_users_active (is_active),
    KEY idx_users_deleted (deleted_at),
    CONSTRAINT fk_users_role FOREIGN KEY (role_id) REFERENCES roles (id)
        ON UPDATE CASCADE
        ON DELETE RESTRICT,
    CONSTRAINT fk_users_queue FOREIGN KEY (queue_id) REFERENCES queues (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE priorities (
    id TINYINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(50) NOT NULL,
    name VARCHAR(100) NOT NULL,
    response_target_minutes SMALLINT UNSIGNED DEFAULT 0,
    resolution_target_minutes SMALLINT UNSIGNED DEFAULT 0,
    color VARCHAR(20) DEFAULT NULL,
    sort_order TINYINT UNSIGNED DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_priorities_slug (slug)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE statuses (
    id TINYINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(50) NOT NULL,
    name VARCHAR(100) NOT NULL,
    description VARCHAR(255) DEFAULT NULL,
    is_terminal TINYINT(1) NOT NULL DEFAULT 0,
    pauses_sla TINYINT(1) NOT NULL DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_statuses_slug (slug)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE categories (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    parent_id INT UNSIGNED DEFAULT NULL,
    name VARCHAR(120) NOT NULL,
    slug VARCHAR(120) NOT NULL,
    description VARCHAR(255) DEFAULT NULL,
    queue_id INT UNSIGNED DEFAULT NULL,
    sort_order SMALLINT UNSIGNED DEFAULT 0,
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_categories_slug (slug),
    KEY idx_categories_parent (parent_id),
    KEY idx_categories_queue (queue_id),
    KEY idx_categories_active (is_active),
    CONSTRAINT fk_categories_parent FOREIGN KEY (parent_id) REFERENCES categories (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL,
    CONSTRAINT fk_categories_queue FOREIGN KEY (queue_id) REFERENCES queues (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE ticket_fields (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    category_id INT UNSIGNED NOT NULL,
    name VARCHAR(120) NOT NULL,
    label VARCHAR(150) NOT NULL,
    type ENUM('text', 'textarea', 'select', 'number', 'boolean') NOT NULL DEFAULT 'text',
    options TEXT DEFAULT NULL,
    is_required TINYINT(1) NOT NULL DEFAULT 0,
    sort_order TINYINT UNSIGNED DEFAULT 0,
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_ticket_fields_category_name (category_id, name),
    KEY idx_ticket_fields_category (category_id),
    CONSTRAINT fk_ticket_fields_category FOREIGN KEY (category_id) REFERENCES categories (id)
        ON UPDATE CASCADE
        ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE sla_policies (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    category_id INT UNSIGNED DEFAULT NULL,
    priority_id TINYINT UNSIGNED DEFAULT NULL,
    client_type ENUM('interno', 'externo') DEFAULT NULL,
    response_minutes SMALLINT UNSIGNED NOT NULL,
    resolution_minutes SMALLINT UNSIGNED NOT NULL,
    sort_order SMALLINT UNSIGNED DEFAULT 0,
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    KEY idx_sla_category (category_id),
    KEY idx_sla_priority (priority_id),
    KEY idx_sla_client_type (client_type),
    CONSTRAINT fk_sla_category FOREIGN KEY (category_id) REFERENCES categories (id)
        ON UPDATE CASCADE
        ON DELETE CASCADE,
    CONSTRAINT fk_sla_priority FOREIGN KEY (priority_id) REFERENCES priorities (id)
        ON UPDATE CASCADE
        ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE slas (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(150) NOT NULL,
    description TEXT DEFAULT NULL,
    first_response_minutes INT UNSIGNED NOT NULL,
    resolution_minutes INT UNSIGNED NOT NULL,
    business_hours_only TINYINT(1) NOT NULL DEFAULT 0,
    escalation_enabled TINYINT(1) NOT NULL DEFAULT 0,
    escalation_minutes INT UNSIGNED DEFAULT NULL,
    priority_id TINYINT UNSIGNED DEFAULT NULL,
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at DATETIME DEFAULT NULL,
    UNIQUE KEY uq_slas_name (name),
    KEY idx_slas_priority (priority_id),
    KEY idx_slas_active (is_active),
    KEY idx_slas_deleted (deleted_at),
    CONSTRAINT fk_slas_priority FOREIGN KEY (priority_id) REFERENCES priorities (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE tickets (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    reference VARCHAR(20) NOT NULL,
    client_type ENUM('interno', 'externo') NOT NULL DEFAULT 'interno',
    main_category VARCHAR(50) NOT NULL DEFAULT 'ti',
    site_name VARCHAR(150) DEFAULT NULL,
    unit_name VARCHAR(150) DEFAULT NULL,
    condominium VARCHAR(150) DEFAULT NULL,
    customer_name VARCHAR(150) DEFAULT NULL,
    contact_name VARCHAR(150) DEFAULT NULL,
    contact_phone VARCHAR(60) DEFAULT NULL,
    contact_email VARCHAR(150) DEFAULT NULL,
    subject VARCHAR(255) NOT NULL,
    description MEDIUMTEXT NOT NULL,
    requester_id BIGINT UNSIGNED NOT NULL,
    assignee_id BIGINT UNSIGNED DEFAULT NULL,
    queue_id INT UNSIGNED NOT NULL,
    category_id INT UNSIGNED DEFAULT NULL,
    priority_id TINYINT UNSIGNED NOT NULL,
    status_id TINYINT UNSIGNED NOT NULL,
    sla_due_at DATETIME DEFAULT NULL,
    resolved_at DATETIME DEFAULT NULL,
    closed_at DATETIME DEFAULT NULL,
    customer_visible TINYINT(1) NOT NULL DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_tickets_reference (reference),
    KEY idx_tickets_client_type (client_type),
    KEY idx_tickets_main_category (main_category),
    KEY idx_tickets_status (status_id),
    KEY idx_tickets_priority (priority_id),
    KEY idx_tickets_queue (queue_id),
    KEY idx_tickets_category (category_id),
    KEY idx_tickets_requester (requester_id),
    KEY idx_tickets_assignee (assignee_id),
    CONSTRAINT fk_tickets_requester FOREIGN KEY (requester_id) REFERENCES users (id)
        ON UPDATE CASCADE
        ON DELETE RESTRICT,
    CONSTRAINT fk_tickets_assignee FOREIGN KEY (assignee_id) REFERENCES users (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL,
    CONSTRAINT fk_tickets_queue FOREIGN KEY (queue_id) REFERENCES queues (id)
        ON UPDATE CASCADE
        ON DELETE RESTRICT,
    CONSTRAINT fk_tickets_category FOREIGN KEY (category_id) REFERENCES categories (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL,
    CONSTRAINT fk_tickets_priority FOREIGN KEY (priority_id) REFERENCES priorities (id)
        ON UPDATE CASCADE
        ON DELETE RESTRICT,
    CONSTRAINT fk_tickets_status FOREIGN KEY (status_id) REFERENCES statuses (id)
        ON UPDATE CASCADE
        ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE ticket_comments (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ticket_id BIGINT UNSIGNED NOT NULL,
    author_id BIGINT UNSIGNED DEFAULT NULL,
    content MEDIUMTEXT NOT NULL,
    is_internal TINYINT(1) NOT NULL DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    KEY idx_ticket_comments_ticket (ticket_id),
    KEY idx_ticket_comments_author (author_id),
    CONSTRAINT fk_ticket_comments_ticket FOREIGN KEY (ticket_id) REFERENCES tickets (id)
        ON UPDATE CASCADE
        ON DELETE CASCADE,
    CONSTRAINT fk_ticket_comments_author FOREIGN KEY (author_id) REFERENCES users (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE ticket_emergency_actions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ticket_id BIGINT UNSIGNED NOT NULL,
    action_type VARCHAR(60) NOT NULL,
    authority VARCHAR(120) NOT NULL,
    contact_name VARCHAR(150) DEFAULT NULL,
    contact_phone VARCHAR(60) DEFAULT NULL,
    protocol_code VARCHAR(120) DEFAULT NULL,
    occurred_at DATETIME NOT NULL,
    notes VARCHAR(255) DEFAULT NULL,
    created_by BIGINT UNSIGNED DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    KEY idx_emergency_ticket (ticket_id),
    KEY idx_emergency_occurred_at (occurred_at),
    CONSTRAINT fk_emergency_ticket FOREIGN KEY (ticket_id) REFERENCES tickets (id)
        ON UPDATE CASCADE
        ON DELETE CASCADE,
    CONSTRAINT fk_emergency_created_by FOREIGN KEY (created_by) REFERENCES users (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE ticket_field_values (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ticket_id BIGINT UNSIGNED NOT NULL,
    field_id INT UNSIGNED NOT NULL,
    value TEXT DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_ticket_field_values (ticket_id, field_id),
    KEY idx_ticket_field_values_field (field_id),
    CONSTRAINT fk_ticket_field_values_ticket FOREIGN KEY (ticket_id) REFERENCES tickets (id)
        ON UPDATE CASCADE
        ON DELETE CASCADE,
    CONSTRAINT fk_ticket_field_values_field FOREIGN KEY (field_id) REFERENCES ticket_fields (id)
        ON UPDATE CASCADE
        ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE notification_outbox (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ticket_id BIGINT UNSIGNED DEFAULT NULL,
    event_type VARCHAR(60) NOT NULL,
    recipient_email VARCHAR(150) NOT NULL,
    recipient_name VARCHAR(150) DEFAULT NULL,
    subject VARCHAR(255) NOT NULL,
    body TEXT NOT NULL,
    status ENUM('pending', 'sent', 'error') NOT NULL DEFAULT 'pending',
    attempts TINYINT UNSIGNED NOT NULL DEFAULT 0,
    last_error VARCHAR(255) DEFAULT NULL,
    sent_at DATETIME DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    KEY idx_notification_status (status),
    KEY idx_notification_ticket (ticket_id),
    CONSTRAINT fk_notification_ticket FOREIGN KEY (ticket_id) REFERENCES tickets (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE admin_audit_logs (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED DEFAULT NULL,
    entity VARCHAR(80) NOT NULL,
    action VARCHAR(60) NOT NULL,
    entity_id BIGINT UNSIGNED DEFAULT NULL,
    payload JSON DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    KEY idx_audit_user (user_id),
    KEY idx_audit_entity (entity),
    CONSTRAINT fk_audit_user FOREIGN KEY (user_id) REFERENCES users (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE ticket_attachments (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ticket_id BIGINT UNSIGNED NOT NULL,
    comment_id BIGINT UNSIGNED DEFAULT NULL,
    filename VARCHAR(255) NOT NULL,
    path VARCHAR(255) NOT NULL,
    mime_type VARCHAR(120) DEFAULT NULL,
    file_size INT UNSIGNED DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    KEY idx_ticket_attachments_ticket (ticket_id),
    KEY idx_ticket_attachments_comment (comment_id),
    CONSTRAINT fk_ticket_attachments_ticket FOREIGN KEY (ticket_id) REFERENCES tickets (id)
        ON UPDATE CASCADE
        ON DELETE CASCADE,
    CONSTRAINT fk_ticket_attachments_comment FOREIGN KEY (comment_id) REFERENCES ticket_comments (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE ticket_status_history (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ticket_id BIGINT UNSIGNED NOT NULL,
    from_status_id TINYINT UNSIGNED DEFAULT NULL,
    to_status_id TINYINT UNSIGNED NOT NULL,
    changed_by BIGINT UNSIGNED DEFAULT NULL,
    notes VARCHAR(255) DEFAULT NULL,
    changed_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    KEY idx_status_history_ticket (ticket_id),
    KEY idx_status_history_to_status (to_status_id),
    KEY idx_status_history_changed_by (changed_by),
    CONSTRAINT fk_status_history_ticket FOREIGN KEY (ticket_id) REFERENCES tickets (id)
        ON UPDATE CASCADE
        ON DELETE CASCADE,
    CONSTRAINT fk_status_history_from_status FOREIGN KEY (from_status_id) REFERENCES statuses (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL,
    CONSTRAINT fk_status_history_to_status FOREIGN KEY (to_status_id) REFERENCES statuses (id)
        ON UPDATE CASCADE
        ON DELETE RESTRICT,
    CONSTRAINT fk_status_history_changed_by FOREIGN KEY (changed_by) REFERENCES users (id)
        ON UPDATE CASCADE
        ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
