(function () {
    'use strict';

    var normalizeHexValue = function (value) {
        if (!value) {
            return null;
        }
        var match = value.trim().match(/^#?([0-9A-Fa-f]{6})$/);
        return match ? '#' + match[1].toUpperCase() : null;
    };

    var setInvalidState = function (input, isInvalid) {
        if (!input) {
            return;
        }
        input.classList.toggle('is-invalid', Boolean(isInvalid));
        if (isInvalid) {
            input.setAttribute('aria-invalid', 'true');
        } else {
            input.removeAttribute('aria-invalid');
        }
    };

    var initColorPickers = function () {
        document.querySelectorAll('[data-color-picker-field]').forEach(function (field) {
            var nativeInput = field.querySelector('[data-color-native]');
            var hexInput = field.querySelector('[data-color-hex]');
            var preview = field.querySelector('[data-color-preview]');
            var currentColor = normalizeHexValue(hexInput ? hexInput.value : '') || normalizeHexValue(nativeInput ? nativeInput.value : '') || '#000000';

            var applyColor = function (value, options) {
                var normalized = normalizeHexValue(value);
                if (!normalized) {
                    return;
                }
                currentColor = normalized;
                var skipNativeUpdate = options && options.skipNative;
                if (nativeInput && !skipNativeUpdate) {
                    nativeInput.value = normalized;
                }
                if (hexInput) {
                    hexInput.value = normalized;
                }
                if (preview) {
                    preview.style.backgroundColor = normalized;
                }
                setInvalidState(hexInput, false);
            };

            if (nativeInput) {
                var syncFromNative = function () {
                    applyColor(nativeInput.value, { skipNative: true });
                };
                nativeInput.addEventListener('input', syncFromNative);
                nativeInput.addEventListener('change', syncFromNative);
            }

            if (hexInput) {
                hexInput.addEventListener('input', function () {
                    var normalized = normalizeHexValue(hexInput.value);
                    if (normalized) {
                        applyColor(normalized);
                    } else {
                        setInvalidState(hexInput, hexInput.value.trim().length > 0);
                    }
                });

                hexInput.addEventListener('blur', function () {
                    if (!normalizeHexValue(hexInput.value)) {
                        hexInput.value = currentColor;
                        setInvalidState(hexInput, false);
                    }
                });
            }

            applyColor(currentColor);
        });
    };

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initColorPickers);
    } else {
        initColorPickers();
    }
})();
