<?php

declare(strict_types=1);

use App\Controllers\AdminController;
use App\Controllers\AdminCategoriesController;
use App\Controllers\AdminCustomerTypesController;
use App\Controllers\AdminFieldsController;
use App\Controllers\AdminQueuesController;
use App\Controllers\AdminSlasController;
use App\Controllers\AdminUsersController;
use App\Controllers\AttachmentController;
use App\Controllers\AuthController;
use App\Controllers\DashboardController;
use App\Controllers\ReportController;
use App\Controllers\LogController;
use App\Controllers\StatusController;
use App\Controllers\TicketController;
use App\Middlewares\AuthMiddleware;
use App\Middlewares\PermissionMiddleware;
use App\Repositories\AdminAuditRepository;
use App\Repositories\AttachmentRepository;
use App\Repositories\CategoryRepository;
use App\Repositories\CommentRepository;
use App\Repositories\CustomerTypeRepository;
use App\Repositories\EmergencyActionRepository;
use App\Repositories\LookupRepository;
use App\Repositories\NotificationOutboxRepository;
use App\Repositories\PriorityRepository;
use App\Repositories\QueueRepository;
use App\Repositories\ReportRepository;
use App\Repositories\RoleRepository;
use App\Repositories\SlaPolicyRepository;
use App\Repositories\SlaRepository;
use App\Repositories\ThemeRepository;
use App\Repositories\TicketFieldRepository;
use App\Repositories\TicketRepository;
use App\Repositories\UserRepository;
use App\Services\AdminService;
use App\Services\AttachmentService;
use App\Services\AuthService;
use App\Services\CategoryService;
use App\Services\CustomerTypeService;
use App\Services\FieldService;
use App\Services\NotificationService;
use App\Services\QueueService;
use App\Services\SlaService;
use App\Services\LogService;
use App\Services\SsoTokenService;
use App\Services\ThemeService;
use App\Services\UserService;
use App\Services\TicketService;

/**
 * Front controller for the HelpDesk mini framework.
 * Provides minimal routing with basic safety.
 */

// Ensure errors are displayed during development only.
ini_set('display_errors', '1');
error_reporting(E_ALL);

// Define path constants for future includes.
define('BASE_PATH', dirname(__DIR__));
define('APP_PATH', BASE_PATH . '/app');

spl_autoload_register(static function (string $class): void {
    $prefix = 'App\\';
    if (!str_starts_with($class, $prefix)) {
        return;
    }

    $relative = str_replace('\\', '/', substr($class, strlen($prefix)));
    $file = APP_PATH . '/' . $relative . '.php';

    if (file_exists($file)) {
        require_once $file;
    }
});

require_once APP_PATH . '/Config/database.php';
require_once APP_PATH . '/Helpers/auth.php';
require_once APP_PATH . '/Helpers/theme.php';
require_once APP_PATH . '/Helpers/url.php';
require_once APP_PATH . '/Helpers/time.php';
require_once APP_PATH . '/Helpers/security.php';
require_once APP_PATH . '/Helpers/logger.php';

if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}

header('X-Frame-Options: SAMEORIGIN');
header('X-Content-Type-Options: nosniff');
header('Referrer-Policy: same-origin');
header('Permissions-Policy: camera=(), microphone=()');
header("Content-Security-Policy: default-src 'self'; img-src 'self' data:; script-src 'self'; style-src 'self' 'unsafe-inline' https://fonts.googleapis.com; font-src 'self' https://fonts.gstatic.com; connect-src 'self';");

/**
 * Simple Router supporting GET/POST dispatch.
 */
final class Router
{
    private string $baseDir;
    private string $entryPoint;

    public function __construct(string $baseDir = '', string $entryPoint = '')
    {
        $baseDir = rtrim($baseDir, '/');
        if ($baseDir === '.' || $baseDir === '') {
            $baseDir = '';
        }
        $this->baseDir = $baseDir;
        $this->entryPoint = rtrim($entryPoint, '/');
    }

    /**
     * @var array<string, callable>
     */
    private array $routes = [
        'GET' => [],
        'POST' => [],
    ];

    public function get(string $path, callable $handler): void
    {
        $this->addRoute('GET', $path, $handler);
    }

    public function post(string $path, callable $handler): void
    {
        $this->addRoute('POST', $path, $handler);
    }

    private function addRoute(string $method, string $path, callable $handler): void
    {
        $normalized = $this->normalizePath($path);
        $this->routes[$method][$normalized] = $handler;
    }

    public function dispatch(string $method, string $uri): void
    {
        $path = $this->normalizePath(parse_url($uri, PHP_URL_PATH) ?? '/');
        $path = $this->stripBasePath($path);
        $handler = $this->routes[$method][$path] ?? null;

        if ($handler === null) {
            $this->respondNotFound();
            return;
        }

        $response = $handler();

        if (is_array($response)) {
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            return;
        }

        echo (string) $response;
    }

    private function normalizePath(string $path): string
    {
        $trimmed = '/' . trim($path, '/');
        return $trimmed === '//' ? '/' : $trimmed;
    }

    private function respondNotFound(): void
    {
        http_response_code(404);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'error' => 'Not Found',
            'message' => 'Route not defined for this path.',
        ]);
    }

    private function stripBasePath(string $path): string
    {
        $candidates = array_filter([
            $this->entryPoint,
            $this->baseDir,
        ]);

        foreach ($candidates as $candidate) {
            if ($candidate === '') {
                continue;
            }

            if ($path === $candidate) {
                return '/';
            }

            $prefix = $candidate . '/';
            if (str_starts_with($path, $prefix)) {
                $trimmed = substr($path, strlen($candidate));
                return $trimmed === '' ? '/' : $this->normalizePath($trimmed);
            }
        }

        return $path;
    }
}

$scriptName = str_replace('\\', '/', $_SERVER['SCRIPT_NAME'] ?? '/index.php');
if ($scriptName === '') {
    $scriptName = '/index.php';
}
$scriptDir = rtrim(str_replace('\\', '/', dirname($scriptName)), '/');
if ($scriptDir === '.' || $scriptDir === '/') {
    $scriptDir = '';
}
$entryPoint = rtrim($scriptName, '/');
$GLOBALS['app.base_dir'] = $scriptDir;
$GLOBALS['app.entry_point'] = $entryPoint;
$router = new Router($scriptDir, $entryPoint);

$userRepository = new UserRepository();
$ticketRepository = new TicketRepository();
$commentRepository = new CommentRepository();
$ticketFieldRepository = new TicketFieldRepository();
$attachmentRepository = new AttachmentRepository();
$priorityRepository = new PriorityRepository();
$queueRepository = new QueueRepository();
$notificationOutboxRepository = new NotificationOutboxRepository();
$slaPolicyRepository = new SlaPolicyRepository();
$slaRepository = new SlaRepository();
$themeRepository = new ThemeRepository();
$lookupRepository = new LookupRepository();
$reportRepository = new ReportRepository();
$roleRepository = new RoleRepository();
$adminAuditRepository = new AdminAuditRepository();
$GLOBALS['audit.repo'] = $adminAuditRepository;
$emergencyActionRepository = new EmergencyActionRepository();
$customerTypeRepository = new CustomerTypeRepository();
$authService = new AuthService($userRepository);
$attachmentService = new AttachmentService($attachmentRepository);
$slaService = new SlaService($slaPolicyRepository, $priorityRepository, $slaRepository);
$notificationService = new NotificationService($notificationOutboxRepository);
$categoryRepository = new CategoryRepository();
$ticketService = new TicketService(
    $ticketRepository,
    $commentRepository,
    $attachmentService,
    $ticketFieldRepository,
    $categoryRepository,
    $customerTypeRepository,
    $emergencyActionRepository,
    $slaService,
    $notificationService,
    $queueRepository,
    $userRepository
);
$categoryService = new CategoryService($categoryRepository, $queueRepository, $adminAuditRepository);
$customerTypeService = new CustomerTypeService($customerTypeRepository, $adminAuditRepository);
$queueService = new QueueService($queueRepository, $adminAuditRepository);
$fieldService = new FieldService($ticketFieldRepository, $categoryRepository, $adminAuditRepository);
$logService = new LogService(
    $adminAuditRepository,
    BASE_PATH . '/storage/logs/app.log',
    BASE_PATH . '/storage/logs/database.log'
);
$adminService = new AdminService(
    $slaPolicyRepository,
    $themeRepository,
    $userRepository,
    $adminAuditRepository,
    $customerTypeRepository
);
$userService = new UserService($userRepository, $adminAuditRepository);
$themeService = new ThemeService($themeRepository);
$ssoTokenService = new SsoTokenService();
$authController = new AuthController($authService, $ssoTokenService);
$dashboardController = new DashboardController($authService);
$ticketController = new TicketController($authService, $ticketService, $lookupRepository);
$adminController = new AdminController(
    $authService,
    $adminService,
    $categoryRepository,
    $categoryService,
    $slaPolicyRepository,
    $themeRepository,
    $queueRepository,
    $roleRepository,
    $lookupRepository,
    $adminAuditRepository,
    $fieldService,
    $queueService
);
$adminCategoriesController = new AdminCategoriesController(
    $authService,
    $categoryService
);
$adminQueuesController = new AdminQueuesController(
    $authService,
    $queueService
);
$adminCustomerTypesController = new AdminCustomerTypesController(
    $authService,
    $customerTypeService
);
$adminFieldsController = new AdminFieldsController(
    $authService,
    $fieldService,
    $categoryRepository
);
$adminUsersController = new AdminUsersController(
    $authService,
    $userService,
    $roleRepository,
    $queueRepository
);
$adminSlasController = new AdminSlasController(
    $authService,
    $slaService,
    $priorityRepository
);
$reportController = new ReportController($authService, $reportRepository);
$attachmentController = new AttachmentController($authService, $ticketService, $attachmentService);
$logController = new LogController($authService, $logService);
$statusController = new StatusController($notificationOutboxRepository);
$authMiddleware = new AuthMiddleware($authService);
$permissionMiddleware = new PermissionMiddleware($authService);

$GLOBALS['app.theme_variables'] = $themeService->variables();
$GLOBALS['app.theme_brand'] = $themeService->brand();
$GLOBALS['app.theme_logo'] = $themeService->logo();

$router->get('/health', static function (): array {
    return [
        'status' => 'ok',
        'timestamp' => (new DateTimeImmutable('now', new DateTimeZone('UTC')))->format(DateTimeInterface::ATOM),
        'service' => 'helpdesk-api',
    ];
});

$router->get('/status', static function () use ($statusController): string {
    return $statusController->index();
});

$router->get('/', static function () use ($authService): void {
    $target = $authService->check() ? 'dashboard' : 'login';
    header('Location: ' . app_url($target));
    exit;
});

$router->get('/login', static function () use ($authController): string {
    return $authController->showLogin();
});

$router->post('/login', static function () use ($authController): string {
    $authController->handleLogin();
    return '';
});

$router->get('/logout', static function () use ($authController): string {
    $authController->logout();
    return '';
});

$router->get('/sso', static function () use ($authController): string {
    $authController->handleSso();
    return '';
});

$router->get('/dashboard', static function () use ($authMiddleware, $permissionMiddleware, $dashboardController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'view_dashboard');
    return $dashboardController->index();
});

$router->get('/tickets', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'view_tickets');
    return $ticketController->index();
});

$router->get('/tickets/minha-fila', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'view_tickets');
    return $ticketController->myQueue();
});

$router->get('/tickets/show', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'view_tickets');
    return $ticketController->show();
});

$router->get('/tickets/create', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'create_ticket');
    return $ticketController->create();
});

$router->post('/tickets/store', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'create_ticket');
    $ticketController->store();
    return '';
});

$router->get('/tickets/edit', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'update_ticket');
    return $ticketController->edit();
});

$router->post('/tickets/update', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'update_ticket');
    $ticketController->update();
    return '';
});

$router->post('/tickets/delete', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'delete_ticket');
    $ticketController->destroy();
    return '';
});

$router->post('/tickets/comment', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'view_tickets');
    $ticketController->comment();
    return '';
});

$router->post('/tickets/emergency-log', static function () use ($authMiddleware, $permissionMiddleware, $ticketController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'update_ticket');
    $ticketController->logEmergencyAction();
    return '';
});

$router->get('/admin', static function () use ($authMiddleware, $permissionMiddleware, $adminController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminController->index();
});

$router->get('/admin/categories', static function () use ($authMiddleware, $permissionMiddleware, $adminCategoriesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminCategoriesController->index();
});

$router->get('/admin/categories/create', static function () use ($authMiddleware, $permissionMiddleware, $adminCategoriesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminCategoriesController->create();
});

$router->post('/admin/categories/store', static function () use ($authMiddleware, $permissionMiddleware, $adminCategoriesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminCategoriesController->store();
    return '';
});

$router->get('/admin/categories/edit', static function () use ($authMiddleware, $permissionMiddleware, $adminCategoriesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminCategoriesController->edit();
});

$router->post('/admin/categories/update', static function () use ($authMiddleware, $permissionMiddleware, $adminCategoriesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminCategoriesController->update();
    return '';
});

$router->get('/admin/categories/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminCategoriesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminCategoriesController->confirmDelete();
});

$router->post('/admin/categories/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminCategoriesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminCategoriesController->delete();
    return '';
});

$router->post('/admin/categories/toggle', static function () use ($authMiddleware, $permissionMiddleware, $adminCategoriesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminCategoriesController->toggleStatus();
    return '';
});

$router->get('/admin/customer-types', static function () use ($authMiddleware, $permissionMiddleware, $adminCustomerTypesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminCustomerTypesController->index();
});

$router->get('/admin/customer-types/create', static function () use ($authMiddleware, $permissionMiddleware, $adminCustomerTypesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminCustomerTypesController->create();
});

$router->post('/admin/customer-types/store', static function () use ($authMiddleware, $permissionMiddleware, $adminCustomerTypesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminCustomerTypesController->store();
    return '';
});

$router->get('/admin/customer-types/edit', static function () use ($authMiddleware, $permissionMiddleware, $adminCustomerTypesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminCustomerTypesController->edit();
});

$router->post('/admin/customer-types/update', static function () use ($authMiddleware, $permissionMiddleware, $adminCustomerTypesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminCustomerTypesController->update();
    return '';
});

$router->get('/admin/customer-types/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminCustomerTypesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminCustomerTypesController->confirmDelete();
});

$router->post('/admin/customer-types/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminCustomerTypesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminCustomerTypesController->delete();
    return '';
});

$router->post('/admin/customer-types/toggle', static function () use ($authMiddleware, $permissionMiddleware, $adminCustomerTypesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminCustomerTypesController->toggleStatus();
    return '';
});

$router->get('/admin/queues', static function () use ($authMiddleware, $permissionMiddleware, $adminQueuesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminQueuesController->index();
});

$router->get('/admin/queues/create', static function () use ($authMiddleware, $permissionMiddleware, $adminQueuesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminQueuesController->create();
});

$router->post('/admin/queues/store', static function () use ($authMiddleware, $permissionMiddleware, $adminQueuesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminQueuesController->store();
    return '';
});

$router->get('/admin/queues/edit', static function () use ($authMiddleware, $permissionMiddleware, $adminQueuesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminQueuesController->edit();
});

$router->post('/admin/queues/update', static function () use ($authMiddleware, $permissionMiddleware, $adminQueuesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminQueuesController->update();
    return '';
});

$router->get('/admin/queues/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminQueuesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminQueuesController->confirmDelete();
});

$router->post('/admin/queues/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminQueuesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminQueuesController->delete();
    return '';
});

$router->post('/admin/queues/toggle', static function () use ($authMiddleware, $permissionMiddleware, $adminQueuesController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminQueuesController->toggleStatus();
    return '';
});

$router->get('/admin/fields', static function () use ($authMiddleware, $permissionMiddleware, $adminFieldsController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminFieldsController->index();
});

$router->get('/admin/fields/create', static function () use ($authMiddleware, $permissionMiddleware, $adminFieldsController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminFieldsController->create();
});

$router->post('/admin/fields/store', static function () use ($authMiddleware, $permissionMiddleware, $adminFieldsController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminFieldsController->store();
    return '';
});

$router->get('/admin/fields/edit', static function () use ($authMiddleware, $permissionMiddleware, $adminFieldsController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminFieldsController->edit();
});

$router->post('/admin/fields/update', static function () use ($authMiddleware, $permissionMiddleware, $adminFieldsController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminFieldsController->update();
    return '';
});

$router->get('/admin/fields/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminFieldsController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminFieldsController->confirmDelete();
});

$router->post('/admin/fields/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminFieldsController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminFieldsController->delete();
    return '';
});

$router->post('/admin/fields/toggle', static function () use ($authMiddleware, $permissionMiddleware, $adminFieldsController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminFieldsController->toggleStatus();
    return '';
});

$router->get('/admin/slas', static function () use ($authMiddleware, $permissionMiddleware, $adminSlasController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminSlasController->index();
});

$router->get('/admin/slas/create', static function () use ($authMiddleware, $permissionMiddleware, $adminSlasController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminSlasController->create();
});

$router->post('/admin/slas/store', static function () use ($authMiddleware, $permissionMiddleware, $adminSlasController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminSlasController->store();
    return '';
});

$router->get('/admin/slas/edit', static function () use ($authMiddleware, $permissionMiddleware, $adminSlasController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminSlasController->edit();
});

$router->post('/admin/slas/update', static function () use ($authMiddleware, $permissionMiddleware, $adminSlasController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminSlasController->update();
    return '';
});

$router->get('/admin/slas/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminSlasController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminSlasController->confirmDelete();
});

$router->post('/admin/slas/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminSlasController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminSlasController->delete();
    return '';
});

$router->post('/admin/slas/toggle', static function () use ($authMiddleware, $permissionMiddleware, $adminSlasController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminSlasController->toggleStatus();
    return '';
});

$router->get('/admin/theme', static function () use ($authMiddleware, $permissionMiddleware, $adminController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminController->theme();
});

$router->post('/admin/theme/logo', static function () use ($authMiddleware, $permissionMiddleware, $adminController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminController->uploadThemeLogo();
    return '';
});

$router->post('/admin/theme/logo/remove', static function () use ($authMiddleware, $permissionMiddleware, $adminController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminController->removeThemeLogo();
    return '';
});

$router->get('/admin/users', static function () use ($authMiddleware, $permissionMiddleware, $adminUsersController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminUsersController->index();
});

$router->get('/admin/users/create', static function () use ($authMiddleware, $permissionMiddleware, $adminUsersController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminUsersController->create();
});

$router->post('/admin/users/store', static function () use ($authMiddleware, $permissionMiddleware, $adminUsersController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminUsersController->store();
    return '';
});

$router->get('/admin/users/edit', static function () use ($authMiddleware, $permissionMiddleware, $adminUsersController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminUsersController->edit();
});

$router->post('/admin/users/update', static function () use ($authMiddleware, $permissionMiddleware, $adminUsersController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminUsersController->update();
    return '';
});

$router->get('/admin/users/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminUsersController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $adminUsersController->confirmDelete();
});

$router->post('/admin/users/delete', static function () use ($authMiddleware, $permissionMiddleware, $adminUsersController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminUsersController->delete();
    return '';
});

$router->post('/admin/users/toggle', static function () use ($authMiddleware, $permissionMiddleware, $adminUsersController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminUsersController->toggleStatus();
    return '';
});

$router->post('/admin', static function () use ($authMiddleware, $permissionMiddleware, $adminController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    $adminController->handle();
    return '';
});

$router->get('/admin/logs', static function () use ($authMiddleware, $permissionMiddleware, $logController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'manage_catalog');
    return $logController->index();
});

$router->get('/reports', static function () use ($authMiddleware, $permissionMiddleware, $reportController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle(['gestor']);
    return $reportController->index();
});

$router->get('/reports/export', static function () use ($authMiddleware, $permissionMiddleware, $reportController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle(['gestor']);
    $reportController->export();
    return '';
});

$router->get('/attachments/download', static function () use ($authMiddleware, $permissionMiddleware, $attachmentController): string {
    $authMiddleware->handle();
    $permissionMiddleware->handle([], 'view_tickets');
    $attachmentController->download();
    return '';
});

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$uri = $_SERVER['REQUEST_URI'] ?? '/';

$router->dispatch($method, $uri);
