<?php
declare(strict_types=1);

use App\Config\AppConfig;
use App\Config\Database;
require_once __DIR__ . '/../app/Config/database.php';

session_start();

$errors = [];
$success = false;
$logMessages = [];
$schemaFile = dirname(__DIR__) . '/database/schema.sql';
$seedFile = dirname(__DIR__) . '/database/seed.sql';
$pdo = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $passwordInput = (string)($_POST['password'] ?? '');

    if (!hash_equals(AppConfig::installerPassword(), $passwordInput)) {
        $errors[] = 'Senha de instalacao incorreta.';
    } else {
        try {
            $pdo = Database::connection();
            executeSqlFile($pdo, $schemaFile, $logMessages);
            $pdo->beginTransaction();
            executeSqlFile($pdo, $seedFile, $logMessages);
            $pdo->commit();
            $success = true;
            $_SESSION['installer_authenticated'] = true;
        } catch (\Throwable $exception) {
            if ($pdo instanceof \PDO && $pdo->inTransaction()) {
                $pdo->rollBack();
            }
            $errors[] = 'Falha ao executar instalacao: ' . $exception->getMessage();
        }
    }
}

function executeSqlFile(\PDO $pdo, string $path, array &$logMessages): void
{
    if (!file_exists($path)) {
        throw new \RuntimeException('Arquivo nao encontrado: ' . $path);
    }

    $sql = file_get_contents($path);
    if ($sql === false) {
        throw new \RuntimeException('Nao foi possivel ler o arquivo: ' . $path);
    }

    $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);
    $sql = preg_replace('/^\s*(--|#).*$/m', '', $sql);
    $statements = array_filter(array_map('trim', preg_split('/;\s*(?:\r?\n|$)/', $sql)));

    foreach ($statements as $statement) {
        if ($statement === '') {
            continue;
        }

        $pdo->exec($statement);
        $logMessages[] = 'Executado: ' . substr($statement, 0, 80) . (strlen($statement) > 80 ? '...' : '');
    }
}

?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <title>Instalador HelpDesk</title>
    <style>
        body { font-family: Arial, sans-serif; background: #f5f5f5; margin: 0; padding: 30px; }
        .container { max-width: 640px; margin: 0 auto; background: #fff; padding: 24px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.08); }
        h1 { margin-top: 0; }
        form { margin-top: 16px; }
        input[type="password"] { width: 100%; padding: 12px; margin-bottom: 12px; border: 1px solid #ccc; border-radius: 4px; }
        button { background: #1976d2; color: #fff; border: none; padding: 12px 20px; border-radius: 4px; cursor: pointer; }
        button:hover { background: #0d47a1; }
        .alert { padding: 12px; border-radius: 4px; margin-bottom: 16px; }
        .alert.error { background: #ffebee; color: #c62828; }
        .alert.success { background: #e8f5e9; color: #2e7d32; }
        .log { background: #fafafa; border: 1px solid #eee; padding: 12px; max-height: 200px; overflow-y: auto; font-size: 14px; }
        .credentials { background: #e3f2fd; padding: 12px; border-radius: 4px; margin-top: 16px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>Instalador HelpDesk</h1>
        <p>Informe a senha de instalacao configurada para executar o script de schema e seed.</p>

        <?php if ($success): ?>
            <div class="alert success">Banco instalado com sucesso! O painel ja pode ser acessado com o usuario gestor.</div>
            <div class="credentials">
                <strong>Login Gestor:</strong> gestor@sinarco.com<br>
                <strong>Senha:</strong> admin123
            </div>
            <?php if ($logMessages): ?>
                <div class="log">
                    <?php foreach ($logMessages as $message): ?>
                        <div><?= htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?></div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            <p style="margin-top:16px;">Recomenda-se remover o instalador apos o uso (public/install.php).</p>
        <?php else: ?>
            <?php foreach ($errors as $error): ?>
                <div class="alert error"><?= htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></div>
            <?php endforeach; ?>
            <form method="post">
                <input type="password" name="password" placeholder="Senha de instalacao" required>
                <button type="submit">Instalar</button>
            </form>
        <?php endif; ?>
    </div>
</body>
</html>
