<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\AuthService;
use App\Services\CategoryService;
use RuntimeException;

final class AdminCategoriesController extends Controller
{
    private const OLD_INPUT_KEY = 'admin_categories_old';

    public function __construct(
        private readonly AuthService $authService,
        private readonly CategoryService $categoryService
    ) {
    }

    public function index(): string
    {
        $user = $this->requireGestor();
        $search = isset($_GET['q']) ? (string) $_GET['q'] : null;
        $status = isset($_GET['status']) ? (string) $_GET['status'] : null;
        $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;

        $result = $this->categoryService->getPaginated($search, $status, $page);

        return $this->render('admin/categories/index', array_merge($this->baseData($user, 'categories'), [
            'categories' => $result['data'],
            'pagination' => $result['pagination'],
            'filters' => [
                'q' => $result['filters']['search'] ?? '',
                'status' => $result['filters']['status'] ?? 'all',
            ],
        ]));
    }

    public function create(): string
    {
        $user = $this->requireGestor();

        return $this->render('admin/categories/form', array_merge($this->baseData($user, 'categories'), [
            'formAction' => app_url('admin/categories/store'),
            'formTitle' => 'Nova categoria',
            'isEdit' => false,
            'categoryRecord' => null,
            'parentOptions' => $this->categoryService->parentOptions(),
            'queues' => $this->categoryService->queueOptions(),
        ]));
    }

    public function store(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $input = $_POST ?? [];
        try {
            $this->categoryService->create($input, $actor);
            $this->flash('admin_flash', ['success' => 'Categoria criada com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/categories');
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/categories/create');
        }
    }

    public function edit(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Categoria inválida.']);
            $this->redirect('admin/categories');
        }

        $record = $this->categoryService->findById($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Categoria não encontrada.']);
            $this->redirect('admin/categories');
        }

        return $this->render('admin/categories/form', array_merge($this->baseData($user, 'categories'), [
            'formAction' => app_url('admin/categories/update?id=' . $id),
            'formTitle' => 'Editar categoria',
            'isEdit' => true,
            'categoryRecord' => $record,
            'parentOptions' => $this->categoryService->parentOptions($id, isset($record['parent_id']) ? (int) $record['parent_id'] : null),
            'queues' => $this->categoryService->queueOptions(),
        ]));
    }

    public function update(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Categoria inválida.']);
            $this->redirect('admin/categories');
        }

        $input = $_POST ?? [];
        try {
            $this->categoryService->update($id, $input, $actor);
            $this->flash('admin_flash', ['success' => 'Categoria atualizada com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/categories/edit?id=' . $id);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/categories/edit?id=' . $id);
        }
    }

    public function confirmDelete(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Categoria inválida.']);
            $this->redirect('admin/categories');
        }

        $record = $this->categoryService->findById($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Categoria não encontrada.']);
            $this->redirect('admin/categories');
        }

        return $this->render('admin/categories/delete', array_merge($this->baseData($user, 'categories'), [
            'categoryRecord' => $record,
        ]));
    }

    public function delete(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Categoria inválida.']);
            $this->redirect('admin/categories');
        }

        $redirect = trim((string) ($_POST['redirect'] ?? 'admin/categories'));
        if ($redirect === '') {
            $redirect = 'admin/categories';
        }

        if ($this->categoryService->isInUse($id)) {
            $this->flash('admin_errors', ['Registro em uso. Não pode excluir. Inative.']);
            $this->redirect($redirect);
        }

        try {
            $this->categoryService->delete($id, $actor);
            $this->flash('admin_flash', ['success' => 'Categoria removida com sucesso.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    public function toggleStatus(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        $activate = isset($_POST['status']) && (int) $_POST['status'] === 1;
        $redirect = isset($_POST['redirect']) ? (string) $_POST['redirect'] : 'admin/categories';

        if ($id <= 0) {
            $this->flash('admin_errors', ['Categoria inválida.']);
            $this->redirect($redirect);
        }

        try {
            $this->categoryService->toggleStatus($id, $activate, $actor);
            $this->flash('admin_flash', ['success' => 'Status atualizado.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    private function requireGestor(): array
    {
        $user = $this->authService->user();
        if ($user === null || ($user['role_slug'] ?? '') !== 'gestor') {
            throw new RuntimeException('Acesso restrito aos gestores.');
        }

        return $user;
    }

    private function flash(string $key, array $value): void
    {
        $_SESSION[$key] = $value;
    }

    /**
     * @param array<string, mixed> $user
     * @return array<string, mixed>
     */
    private function baseData(array $user, string $section): array
    {
        return [
            'flash' => $this->pullFlash('admin_flash', []),
            'errors' => $this->pullFlash('admin_errors', []),
            'old' => $this->pullOldInput(),
            'pageTitle' => 'Administração',
            'authUser' => $user,
            'adminSection' => $section,
        ];
    }

    /**
     * @return array<mixed>
     */
    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }

    private function rememberOldInput(array $input): void
    {
        $payload = $input;
        $payload['is_active'] = isset($input['is_active']) ? 1 : 0;
        $_SESSION[self::OLD_INPUT_KEY] = $payload;
    }

    /**
     * @return array<string, mixed>
     */
    private function pullOldInput(): array
    {
        $data = $_SESSION[self::OLD_INPUT_KEY] ?? [];
        unset($_SESSION[self::OLD_INPUT_KEY]);

        return is_array($data) ? $data : [];
    }

    private function clearOldInput(): void
    {
        unset($_SESSION[self::OLD_INPUT_KEY]);
    }

    private function redirect(string $path): void
    {
        header('Location: ' . app_url($path));
        exit;
    }
}
