<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\AuthService;
use App\Services\CustomerTypeService;
use RuntimeException;

final class AdminCustomerTypesController extends Controller
{
    private const OLD_INPUT_KEY = 'admin_customer_types_old';

    public function __construct(
        private readonly AuthService $authService,
        private readonly CustomerTypeService $customerTypes
    ) {
    }

    public function index(): string
    {
        $user = $this->requireGestor();
        $search = isset($_GET['q']) ? (string) $_GET['q'] : null;
        $status = isset($_GET['status']) ? (string) $_GET['status'] : null;
        $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;

        $result = $this->customerTypes->getPaginated($search, $status, $page);

        return $this->render('admin/customer-types/index', array_merge($this->baseData($user, 'customer-types'), [
            'types' => $result['data'],
            'pagination' => $result['pagination'],
            'filters' => [
                'q' => $result['filters']['search'] ?? '',
                'status' => $result['filters']['status'] ?? 'all',
            ],
        ]));
    }

    public function create(): string
    {
        $user = $this->requireGestor();

        return $this->render('admin/customer-types/form', array_merge($this->baseData($user, 'customer-types'), [
            'formAction' => app_url('admin/customer-types/store'),
            'formTitle' => 'Novo tipo de cliente',
            'typeRecord' => null,
            'isEdit' => false,
        ]));
    }

    public function store(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $input = $_POST ?? [];
        try {
            $this->customerTypes->create($input, $actor);
            $this->flash('admin_flash', ['success' => 'Tipo de cliente criado com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/customer-types');
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/customer-types/create');
        }
    }

    public function edit(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Tipo inválido.']);
            $this->redirect('admin/customer-types');
        }

        $record = $this->customerTypes->findById($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Registro não encontrado.']);
            $this->redirect('admin/customer-types');
        }

        return $this->render('admin/customer-types/form', array_merge($this->baseData($user, 'customer-types'), [
            'formAction' => app_url('admin/customer-types/update?id=' . $id),
            'formTitle' => 'Editar tipo de cliente',
            'typeRecord' => $record,
            'isEdit' => true,
        ]));
    }

    public function update(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Tipo inválido.']);
            $this->redirect('admin/customer-types');
        }

        $input = $_POST ?? [];
        try {
            $this->customerTypes->update($id, $input, $actor);
            $this->flash('admin_flash', ['success' => 'Tipo atualizado com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/customer-types/edit?id=' . $id);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/customer-types/edit?id=' . $id);
        }
    }

    public function confirmDelete(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Tipo inválido.']);
            $this->redirect('admin/customer-types');
        }

        $record = $this->customerTypes->findById($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Registro não encontrado.']);
            $this->redirect('admin/customer-types');
        }

        return $this->render('admin/customer-types/delete', array_merge($this->baseData($user, 'customer-types'), [
            'typeRecord' => $record,
        ]));
    }

    public function delete(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Tipo inválido.']);
            $this->redirect('admin/customer-types');
        }

        $redirect = trim((string) ($_POST['redirect'] ?? 'admin/customer-types'));
        if ($redirect === '') {
            $redirect = 'admin/customer-types';
        }

        if ($this->customerTypes->isInUse($id)) {
            $this->flash('admin_errors', ['Registro em uso. Não pode excluir. Inative.']);
            $this->redirect($redirect);
        }

        try {
            $this->customerTypes->delete($id, $actor);
            $this->flash('admin_flash', ['success' => 'Tipo removido com sucesso.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    public function toggleStatus(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        $activate = isset($_POST['status']) && (int) $_POST['status'] === 1;
        $redirect = isset($_POST['redirect']) ? (string) $_POST['redirect'] : 'admin/customer-types';

        if ($id <= 0) {
            $this->flash('admin_errors', ['Tipo inválido.']);
            $this->redirect($redirect);
        }

        try {
            $this->customerTypes->toggleStatus($id, $activate, $actor);
            $this->flash('admin_flash', ['success' => 'Status atualizado.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    private function requireGestor(): array
    {
        $user = $this->authService->user();
        if ($user === null || ($user['role_slug'] ?? '') !== 'gestor') {
            throw new RuntimeException('Acesso restrito aos gestores.');
        }

        return $user;
    }

    private function flash(string $key, array $value): void
    {
        $_SESSION[$key] = $value;
    }

    /**
     * @return array<string, mixed>
     */
    private function baseData(array $user, string $section): array
    {
        return [
            'flash' => $this->pullFlash('admin_flash', []),
            'errors' => $this->pullFlash('admin_errors', []),
            'old' => $this->pullOldInput(),
            'pageTitle' => 'Administração',
            'authUser' => $user,
            'adminSection' => $section,
        ];
    }

    private function rememberOldInput(array $input): void
    {
        $_SESSION[self::OLD_INPUT_KEY] = $input;
    }

    private function pullOldInput(): array
    {
        if (!isset($_SESSION[self::OLD_INPUT_KEY])) {
            return [];
        }

        $old = $_SESSION[self::OLD_INPUT_KEY];
        unset($_SESSION[self::OLD_INPUT_KEY]);

        return is_array($old) ? $old : [];
    }

    private function clearOldInput(): void
    {
        unset($_SESSION[self::OLD_INPUT_KEY]);
    }

    /**
     * @return array<mixed>
     */
    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }

    private function redirect(string $path): void
    {
        header('Location: ' . app_url($path));
        exit;
    }
}
