<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Repositories\CategoryRepository;
use App\Services\AuthService;
use App\Services\FieldService;
use RuntimeException;

final class AdminFieldsController extends Controller
{
    private const OLD_INPUT_KEY = 'admin_fields_old';

    public function __construct(
        private readonly AuthService $authService,
        private readonly FieldService $fieldService,
        private readonly CategoryRepository $categories
    ) {
    }

    public function index(): string
    {
        $user = $this->requireGestor();
        $search = isset($_GET['q']) ? (string) $_GET['q'] : null;
        $status = isset($_GET['status']) ? (string) $_GET['status'] : null;
        $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;

        $result = $this->fieldService->getPaginated($search, $status, $page);

        return $this->render('admin/fields/index', array_merge($this->baseData($user, 'fields'), [
            'fields' => $result['data'],
            'pagination' => $result['pagination'],
            'filters' => [
                'q' => $result['filters']['search'] ?? '',
                'status' => $result['filters']['status'] ?? 'all',
            ],
        ]));
    }

    public function create(): string
    {
        $user = $this->requireGestor();

        return $this->render('admin/fields/form', array_merge($this->baseData($user, 'fields'), [
            'formAction' => app_url('admin/fields/store'),
            'formTitle' => 'Novo campo',
            'isEdit' => false,
            'fieldRecord' => null,
            'categories' => $this->categories->all(),
        ]));
    }

    public function store(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $input = $_POST ?? [];
        try {
            $this->fieldService->create($input, $actor);
            $this->flash('admin_flash', ['success' => 'Campo criado com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/fields');
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/fields/create');
        }
    }

    public function edit(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Campo inválido.']);
            $this->redirect('admin/fields');
        }

        $record = $this->fieldService->findById($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Campo não encontrado.']);
            $this->redirect('admin/fields');
        }

        return $this->render('admin/fields/form', array_merge($this->baseData($user, 'fields'), [
            'formAction' => app_url('admin/fields/update?id=' . $id),
            'formTitle' => 'Editar campo',
            'isEdit' => true,
            'fieldRecord' => $record,
            'categories' => $this->categories->all(),
        ]));
    }

    public function update(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Campo inválido.']);
            $this->redirect('admin/fields');
        }

        $input = $_POST ?? [];
        try {
            $this->fieldService->update($id, $input, $actor);
            $this->flash('admin_flash', ['success' => 'Campo atualizado com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/fields/edit?id=' . $id);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/fields/edit?id=' . $id);
        }
    }

    public function confirmDelete(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Campo inválido.']);
            $this->redirect('admin/fields');
        }

        $record = $this->fieldService->findById($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Campo não encontrado.']);
            $this->redirect('admin/fields');
        }

        return $this->render('admin/fields/delete', array_merge($this->baseData($user, 'fields'), [
            'fieldRecord' => $record,
        ]));
    }

    public function delete(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Campo inválido.']);
            $this->redirect('admin/fields');
        }

        $redirect = trim((string) ($_POST['redirect'] ?? 'admin/fields'));
        if ($redirect === '') {
            $redirect = 'admin/fields';
        }

        if ($this->fieldService->isInUse($id)) {
            $this->flash('admin_errors', ['Registro em uso. Não pode excluir. Inative.']);
            $this->redirect($redirect);
        }

        try {
            $this->fieldService->delete($id, $actor);
            $this->flash('admin_flash', ['success' => 'Campo removido com sucesso.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    public function toggleStatus(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        $activate = isset($_POST['status']) && (int) $_POST['status'] === 1;
        $redirect = isset($_POST['redirect']) ? (string) $_POST['redirect'] : 'admin/fields';

        if ($id <= 0) {
            $this->flash('admin_errors', ['Campo inválido.']);
            $this->redirect($redirect);
        }

        try {
            $this->fieldService->toggleStatus($id, $activate, $actor);
            $this->flash('admin_flash', ['success' => 'Status atualizado.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    private function requireGestor(): array
    {
        $user = $this->authService->user();
        if ($user === null || ($user['role_slug'] ?? '') !== 'gestor') {
            throw new RuntimeException('Acesso restrito aos gestores.');
        }

        return $user;
    }

    private function flash(string $key, array $value): void
    {
        $_SESSION[$key] = $value;
    }

    /**
     * @param array<string, mixed> $user
     * @return array<string, mixed>
     */
    private function baseData(array $user, string $section): array
    {
        return [
            'flash' => $this->pullFlash('admin_flash', []),
            'errors' => $this->pullFlash('admin_errors', []),
            'old' => $this->pullOldInput(),
            'pageTitle' => 'Administração',
            'authUser' => $user,
            'adminSection' => $section,
        ];
    }

    /**
     * @return array<mixed>
     */
    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }

    private function rememberOldInput(array $input): void
    {
        $payload = $input;
        $payload['is_required'] = isset($input['is_required']) ? 1 : 0;
        $payload['is_active'] = isset($input['is_active']) ? 1 : 0;
        $_SESSION[self::OLD_INPUT_KEY] = $payload;
    }

    /**
     * @return array<string, mixed>
     */
    private function pullOldInput(): array
    {
        $data = $_SESSION[self::OLD_INPUT_KEY] ?? [];
        unset($_SESSION[self::OLD_INPUT_KEY]);

        return is_array($data) ? $data : [];
    }

    private function clearOldInput(): void
    {
        unset($_SESSION[self::OLD_INPUT_KEY]);
    }

    private function redirect(string $path): void
    {
        header('Location: ' . app_url($path));
        exit;
    }
}
