<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\AuthService;
use App\Services\QueueService;
use RuntimeException;

final class AdminQueuesController extends Controller
{
    private const OLD_INPUT_KEY = 'admin_queues_old';

    public function __construct(
        private readonly AuthService $authService,
        private readonly QueueService $queueService
    ) {
    }

    public function index(): string
    {
        $user = $this->requireGestor();
        $search = isset($_GET['q']) ? (string) $_GET['q'] : null;
        $status = isset($_GET['status']) ? (string) $_GET['status'] : null;
        $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;

        $result = $this->queueService->getPaginated($search, $status, $page);

        return $this->render('admin/queues/index', array_merge($this->baseData($user, 'queues'), [
            'queues' => $result['data'],
            'pagination' => $result['pagination'],
            'filters' => [
                'q' => $result['filters']['search'] ?? '',
                'status' => $result['filters']['status'] ?? 'all',
            ],
        ]));
    }

    public function create(): string
    {
        $user = $this->requireGestor();

        return $this->render('admin/queues/form', array_merge($this->baseData($user, 'queues'), [
            'formAction' => app_url('admin/queues/store'),
            'formTitle' => 'Nova fila',
            'isEdit' => false,
            'queueRecord' => null,
        ]));
    }

    public function store(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $input = $_POST ?? [];
        try {
            $this->queueService->create($input, $actor);
            $this->flash('admin_flash', ['success' => 'Fila criada com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/queues');
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/queues/create');
        }
    }

    public function edit(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Fila inválida.']);
            $this->redirect('admin/queues');
        }

        $record = $this->queueService->findById($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Fila não encontrada.']);
            $this->redirect('admin/queues');
        }

        return $this->render('admin/queues/form', array_merge($this->baseData($user, 'queues'), [
            'formAction' => app_url('admin/queues/update?id=' . $id),
            'formTitle' => 'Editar fila',
            'isEdit' => true,
            'queueRecord' => $record,
        ]));
    }

    public function update(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Fila inválida.']);
            $this->redirect('admin/queues');
        }

        $input = $_POST ?? [];
        try {
            $this->queueService->update($id, $input, $actor);
            $this->flash('admin_flash', ['success' => 'Fila atualizada com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/queues/edit?id=' . $id);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/queues/edit?id=' . $id);
        }
    }

    public function confirmDelete(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Fila inválida.']);
            $this->redirect('admin/queues');
        }

        $record = $this->queueService->findById($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Fila não encontrada.']);
            $this->redirect('admin/queues');
        }

        return $this->render('admin/queues/delete', array_merge($this->baseData($user, 'queues'), [
            'queueRecord' => $record,
        ]));
    }

    public function delete(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Fila inválida.']);
            $this->redirect('admin/queues');
        }

        $redirect = trim((string) ($_POST['redirect'] ?? 'admin/queues'));
        if ($redirect === '') {
            $redirect = 'admin/queues';
        }

        if ($this->queueService->isInUse($id)) {
            $this->flash('admin_errors', ['Registro em uso. Não pode excluir. Inative.']);
            $this->redirect($redirect);
        }

        try {
            $this->queueService->delete($id, $actor);
            $this->flash('admin_flash', ['success' => 'Fila removida com sucesso.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    public function toggleStatus(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        $activate = isset($_POST['status']) && (int) $_POST['status'] === 1;
        $redirect = isset($_POST['redirect']) ? (string) $_POST['redirect'] : 'admin/queues';

        if ($id <= 0) {
            $this->flash('admin_errors', ['Fila inválida.']);
            $this->redirect($redirect);
        }

        try {
            $this->queueService->toggleStatus($id, $activate, $actor);
            $this->flash('admin_flash', ['success' => 'Status atualizado.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    private function requireGestor(): array
    {
        $user = $this->authService->user();
        if ($user === null || ($user['role_slug'] ?? '') !== 'gestor') {
            throw new RuntimeException('Acesso restrito aos gestores.');
        }

        return $user;
    }

    private function flash(string $key, array $value): void
    {
        $_SESSION[$key] = $value;
    }

    /**
     * @param array<string, mixed> $user
     * @return array<string, mixed>
     */
    private function baseData(array $user, string $section): array
    {
        return [
            'flash' => $this->pullFlash('admin_flash', []),
            'errors' => $this->pullFlash('admin_errors', []),
            'old' => $this->pullOldInput(),
            'pageTitle' => 'Administração',
            'authUser' => $user,
            'adminSection' => $section,
        ];
    }

    /**
     * @return array<mixed>
     */
    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }

    private function rememberOldInput(array $input): void
    {
        $payload = $input;
        $payload['is_active'] = isset($input['is_active']) ? 1 : 0;
        $_SESSION[self::OLD_INPUT_KEY] = $payload;
    }

    /**
     * @return array<string, mixed>
     */
    private function pullOldInput(): array
    {
        $data = $_SESSION[self::OLD_INPUT_KEY] ?? [];
        unset($_SESSION[self::OLD_INPUT_KEY]);

        return is_array($data) ? $data : [];
    }

    private function clearOldInput(): void
    {
        unset($_SESSION[self::OLD_INPUT_KEY]);
    }

    private function redirect(string $path): void
    {
        header('Location: ' . app_url($path));
        exit;
    }
}
