<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Repositories\QueueRepository;
use App\Repositories\RoleRepository;
use App\Services\AuthService;
use App\Services\UserService;
use RuntimeException;

final class AdminUsersController extends Controller
{
    private const OLD_INPUT_KEY = 'admin_users_old';

    public function __construct(
        private readonly AuthService $authService,
        private readonly UserService $userService,
        private readonly RoleRepository $roles,
        private readonly QueueRepository $queues
    ) {
    }

    public function index(): string
    {
        $user = $this->requireGestor();
        $search = isset($_GET['q']) ? (string) $_GET['q'] : null;
        $status = isset($_GET['status']) ? (string) $_GET['status'] : null;
        $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;

        $result = $this->userService->getPaginated($search, $status, $page);

        return $this->render('admin/users/index', array_merge($this->baseData($user, 'users'), [
            'users' => $result['data'],
            'pagination' => $result['pagination'],
            'filters' => [
                'q' => $result['filters']['search'] ?? '',
                'status' => $result['filters']['status'] ?? 'all',
            ],
            'statusOptions' => [
                'all' => 'Todos',
                'active' => 'Ativos',
                'inactive' => 'Inativos',
            ],
        ]));
    }

    public function create(): string
    {
        $user = $this->requireGestor();

        return $this->render('admin/users/form', array_merge($this->baseData($user, 'users'), [
            'formAction' => app_url('admin/users/store'),
            'formTitle' => 'Novo usuário',
            'roles' => $this->roles->all(),
            'queues' => $this->queues->all(),
            'isEdit' => false,
            'userRecord' => null,
        ]));
    }

    public function store(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $input = $_POST ?? [];

        try {
            $this->userService->create($input, $actor);
            $this->flash('admin_flash', ['success' => 'Usuário criado com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/users');
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/users/create');
        }
    }

    public function edit(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Usuário inválido.']);
            $this->redirect('admin/users');
        }

        $record = $this->userService->find($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Usuário não encontrado.']);
            $this->redirect('admin/users');
        }

        return $this->render('admin/users/form', array_merge($this->baseData($user, 'users'), [
            'formAction' => app_url('admin/users/update?id=' . $id),
            'formTitle' => 'Editar usuário',
            'roles' => $this->roles->all(),
            'queues' => $this->queues->all(),
            'isEdit' => true,
            'userRecord' => $record,
        ]));
    }

    public function update(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Usuário inválido.']);
            $this->redirect('admin/users');
        }

        $input = $_POST ?? [];

        try {
            $this->userService->update($id, $input, $actor);
            $this->flash('admin_flash', ['success' => 'Usuário atualizado com sucesso.']);
            $this->clearOldInput();
            $this->redirect('admin/users/edit?id=' . $id);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
            $this->rememberOldInput($input);
            $this->redirect('admin/users/edit?id=' . $id);
        }
    }

    public function confirmDelete(): string
    {
        $user = $this->requireGestor();
        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Usuário inválido.']);
            $this->redirect('admin/users');
        }

        $record = $this->userService->find($id);
        if ($record === null) {
            $this->flash('admin_errors', ['Usuário não encontrado.']);
            $this->redirect('admin/users');
        }

        return $this->render('admin/users/delete', array_merge($this->baseData($user, 'users'), [
            'userRecord' => $record,
        ]));
    }

    public function delete(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        if ($id <= 0) {
            $this->flash('admin_errors', ['Usuário inválido.']);
            $this->redirect('admin/users');
        }

        $redirect = trim((string) ($_POST['redirect'] ?? 'admin/users'));
        if ($redirect === '') {
            $redirect = 'admin/users';
        }

        if ($this->userService->isInUse($id)) {
            $this->flash('admin_errors', ['Registro em uso. Não pode excluir. Inative.']);
            $this->redirect($redirect);
        }

        try {
            $this->userService->delete($id, $actor);
            $this->flash('admin_flash', ['success' => 'Usuário removido com sucesso.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    public function toggleStatus(): void
    {
        $actor = $this->requireGestor();
        csrf_validate();

        $id = (int) ($_GET['id'] ?? 0);
        $activate = isset($_POST['status']) && (int) $_POST['status'] === 1;
        $redirect = isset($_POST['redirect']) ? (string) $_POST['redirect'] : 'admin/users';

        if ($id <= 0) {
            $this->flash('admin_errors', ['Usuário inválido.']);
            $this->redirect($redirect);
        }

        try {
            $this->userService->toggleStatus($id, $activate, $actor);
            $this->flash('admin_flash', ['success' => 'Status atualizado.']);
        } catch (RuntimeException $exception) {
            $this->flash('admin_errors', [$exception->getMessage()]);
        }

        $this->redirect($redirect);
    }

    private function requireGestor(): array
    {
        $user = $this->authService->user();
        if ($user === null || ($user['role_slug'] ?? '') !== 'gestor') {
            throw new RuntimeException('Acesso restrito aos gestores.');
        }

        return $user;
    }

    private function flash(string $key, array $value): void
    {
        $_SESSION[$key] = $value;
    }

    /**
     * @param array<string, mixed> $user
     * @return array<string, mixed>
     */
    private function baseData(array $user, string $section): array
    {
        return [
            'flash' => $this->pullFlash('admin_flash', []),
            'errors' => $this->pullFlash('admin_errors', []),
            'old' => $this->pullOldInput(),
            'pageTitle' => 'Administração',
            'authUser' => $user,
            'adminSection' => $section,
        ];
    }

    /**
     * @return array<mixed>
     */
    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }

    private function rememberOldInput(array $input): void
    {
        $payload = $input;
        $payload['is_active'] = isset($input['is_active']) ? 1 : 0;
        unset($payload['password'], $payload['password_confirmation']);
        $_SESSION[self::OLD_INPUT_KEY] = $payload;
    }

    /**
     * @return array<string, mixed>
     */
    private function pullOldInput(): array
    {
        $data = $_SESSION[self::OLD_INPUT_KEY] ?? [];
        unset($_SESSION[self::OLD_INPUT_KEY]);

        return is_array($data) ? $data : [];
    }

    private function clearOldInput(): void
    {
        unset($_SESSION[self::OLD_INPUT_KEY]);
    }

    private function redirect(string $path): void
    {
        header('Location: ' . app_url($path));
        exit;
    }
}
