<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\AttachmentService;
use App\Services\AuthService;
use App\Services\TicketService;
use RuntimeException;

final class AttachmentController extends Controller
{
    public function __construct(
        private readonly AuthService $authService,
        private readonly TicketService $ticketService,
        private readonly AttachmentService $attachmentService
    ) {
    }

    public function download(): void
    {
        $user = $this->requireUser();
        $attachmentId = (int) ($_GET['id'] ?? 0);

        if ($attachmentId <= 0) {
            $this->respondNotFound();
        }

        $attachment = $this->ticketService->attachmentForUser($attachmentId, $user);
        if ($attachment === null) {
            $this->respondForbidden();
        }

        $inline = isset($_GET['inline']) && (int) $_GET['inline'] === 1;

        try {
            $this->attachmentService->stream($attachment, $inline);
        } catch (RuntimeException $exception) {
            $this->respondNotFound($exception->getMessage());
        }

        exit;
    }

    private function requireUser(): array
    {
        $user = $this->authService->user();
        if ($user === null) {
            http_response_code(401);
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode([
                'error' => 'Unauthorized',
                'message' => 'Sessão expirada. Faça login novamente.',
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        }

        return $user;
    }

    private function respondForbidden(): void
    {
        http_response_code(403);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'error' => 'Forbidden',
            'message' => 'Você não tem acesso a este anexo.',
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }

    private function respondNotFound(string $message = 'Arquivo não encontrado.'): void
    {
        http_response_code(404);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'error' => 'Not Found',
            'message' => $message,
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }
}
