<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\AuthService;
use App\Services\SsoTokenService;
use RuntimeException;

final class AuthController extends Controller
{
    public function __construct(
        private readonly AuthService $authService,
        private readonly SsoTokenService $ssoTokenService
    ) {
    }

    public function showLogin(): string
    {
        if ($this->authService->check()) {
            $this->redirect($this->sanitizeRedirect(
                $_GET['redirect'] ?? null,
                $this->defaultRedirectForUser($this->authService->user())
            ));
        }

        $errors = $this->pullFlash('auth_errors', []);
        $old = $this->pullFlash('auth_old', []);
        $requestedRedirect = $_GET['redirect'] ?? ($old['redirect'] ?? null);
        $redirect = $this->normalizeRedirectPath($requestedRedirect) ?? '';

        return $this->render('auth/login', [
            'errors' => $errors,
            'old' => array_merge(['email' => '', 'redirect' => $redirect], $old),
        ], 'layouts/auth');
    }

    public function handleLogin(): void
    {
        $errors = [];
        try {
            csrf_validate();
        } catch (RuntimeException $exception) {
            $errors[] = $exception->getMessage();
        }

        $blockedSeconds = $this->loginBlockedSeconds();
        if ($blockedSeconds !== null) {
            $errors[] = 'Muitas tentativas de login. Aguarde ' . $blockedSeconds . 's e tente novamente.';
        }

        $email = trim((string) ($_POST['email'] ?? ''));
        $password = (string) ($_POST['password'] ?? '');
        $rawRedirect = $_POST['redirect'] ?? $_GET['redirect'] ?? null;
        $safeRedirectInput = $this->normalizeRedirectPath($rawRedirect) ?? '';

        if ($email === '' || $password === '') {
            $errors[] = 'Informe email e senha.';
        }

        $clientIp = $this->clientIp();

        if ($errors === [] && !$this->authService->attemptLogin($email, $password)) {
            $this->incrementLoginAttempts();
            app_log('WARNING', 'Login falhou', ['email' => $email, 'ip' => $clientIp]);
            audit_log('auth', 'login_failed', null, ['email' => $email, 'ip' => $clientIp]);
            $errors[] = 'Credenciais invalidas ou usuario inativo.';
        }

        if ($errors !== []) {
            $this->flashBack($errors, ['email' => $email, 'redirect' => $safeRedirectInput]);
            header('Location: ' . app_url('login'));
            exit;
        }

        $this->resetLoginAttempts();
        $authUser = $this->authService->user();
        app_log('INFO', 'Login efetuado', ['email' => $authUser['email'] ?? $email, 'ip' => $clientIp]);
        audit_log('auth', 'login_success', $authUser['id'] ?? null, ['ip' => $clientIp]);
        $redirect = $this->sanitizeRedirect($rawRedirect, $this->defaultRedirectForUser($authUser));
        $this->redirect($redirect);
    }

    public function logout(): void
    {
        $this->authService->logout();
        header('Location: ' . app_url('login'));
        exit;
    }

    public function handleSso(): void
    {
        $requestedRedirect = $_GET['redirect'] ?? null;
        $redirect = $this->sanitizeRedirect($requestedRedirect, '/dashboard');
        $token = (string) ($_GET['token'] ?? '');

        try {
            $payload = $this->ssoTokenService->validateToken($token);
            if (!$this->authService->loginByEmail($payload['email'])) {
                throw new RuntimeException('Usuario nao encontrado ou inativo.');
            }

            $user = $this->authService->user();
            app_log('INFO', 'Login via SSO', ['email' => $payload['email'], 'ip' => $this->clientIp()]);
            audit_log('auth', 'sso_login', $user['id'] ?? null, ['email' => $payload['email']]);
            $finalRedirect = $this->sanitizeRedirect($requestedRedirect, $this->defaultRedirectForUser($user));
            $this->redirect($finalRedirect);
        } catch (RuntimeException $exception) {
            app_log('WARNING', 'SSO falhou', ['error' => $exception->getMessage(), 'ip' => $this->clientIp()]);
            $this->flashBack([$exception->getMessage()], ['redirect' => $redirect]);
            header('Location: ' . app_url('login'));
            exit;
        }
    }

    private function flashBack(array $errors, array $old = []): void
    {
        $_SESSION['auth_errors'] = $errors;
        $_SESSION['auth_old'] = $old;
    }

    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }

    private function sanitizeRedirect(?string $target, ?string $fallback = '/dashboard'): string
    {
        $candidate = $this->normalizeRedirectPath($target);
        if ($candidate !== null) {
            return $candidate;
        }

        $fallbackCandidate = $this->normalizeRedirectPath($fallback);

        return $fallbackCandidate ?? '/dashboard';
    }

    private function normalizeRedirectPath(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim($value);
        if ($value === '') {
            return null;
        }

        $path = parse_url($value, PHP_URL_PATH) ?? '';
        $query = parse_url($value, PHP_URL_QUERY);

        if ($path === '' || !str_starts_with($path, '/')) {
            return null;
        }

        if ($path === '/login' || $path === '/logout') {
            return null;
        }

        return $query ? $path . '?' . $query : $path;
    }

    /**
     * Decide default redirect based on the logged user's role.
     */
    private function defaultRedirectForUser(?array $user): string
    {
        $fallback = '/tickets/minha-fila';
        if ($user === null) {
            return $fallback;
        }

        $slug = strtolower((string) ($user['role_slug'] ?? ''));
        if ($slug === '') {
            $slug = $this->normalizeRoleName($user['role_name'] ?? '');
        }

        return match ($slug) {
            'gestor' => '/reports',
            'atendente-ti', 'atendenteti' => '/tickets/minha-fila',
            'externo', 'solicitanteexterno' => '/tickets',
            default => $fallback,
        };
    }

    private function normalizeRoleName(string $value): string
    {
        $value = strtolower(trim($value));
        $value = preg_replace('/[^a-z0-9]+/i', '', $value) ?? '';

        return $value;
    }

    private function redirect(string $path): void
    {
        header('Location: ' . app_url($path));
        exit;
    }

    private function loginRateState(): array
    {
        $state = $_SESSION['login_rate'] ?? [
            'attempts' => 0,
            'window_start' => time(),
            'blocked_until' => 0,
        ];

        $now = time();
        if ($now - ($state['window_start'] ?? 0) > 60) {
            $state['attempts'] = 0;
            $state['window_start'] = $now;
        }

        $_SESSION['login_rate'] = $state;

        return $state;
    }

    private function loginBlockedSeconds(): ?int
    {
        $state = $this->loginRateState();
        $blockedUntil = (int) ($state['blocked_until'] ?? 0);

        if ($blockedUntil > time()) {
            return $blockedUntil - time();
        }

        return null;
    }

    private function incrementLoginAttempts(): void
    {
        $state = $this->loginRateState();
        $state['attempts'] = (int) ($state['attempts'] ?? 0) + 1;

        if ($state['attempts'] >= 5) {
            $state['blocked_until'] = time() + 60;
            $state['attempts'] = 0;
            $state['window_start'] = time();
        }

        $_SESSION['login_rate'] = $state;
    }

    private function resetLoginAttempts(): void
    {
        $_SESSION['login_rate'] = [
            'attempts' => 0,
            'window_start' => time(),
            'blocked_until' => 0,
        ];
    }

    private function clientIp(): string
    {
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
}
