<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\AuthService;
use App\Services\LogService;
use DateTimeImmutable;
use RuntimeException;

final class LogController extends Controller
{
    private const TYPE_LABELS = [
        'application' => 'Log da aplicação',
        'pdo' => 'Log do banco/PDO',
        'audit' => 'Audit Log detalhado',
    ];

    private const MAX_RANGE_DAYS = 31;

    public function __construct(
        private readonly AuthService $authService,
        private readonly LogService $logService
    ) {
    }

    public function index(): string
    {
        $user = $this->requireGestor();
        $type = isset($_GET['type']) ? (string) $_GET['type'] : '';
        $start = isset($_GET['start']) ? (string) $_GET['start'] : '';
        $end = isset($_GET['end']) ? (string) $_GET['end'] : '';
        $page = isset($_GET['page']) ? max(1, (int) $_GET['page']) : 1;

        $filters = [
            'type' => $type,
            'start' => $start,
            'end' => $end,
        ];

        $errors = [];
        $report = null;

        if ($type !== '') {
            if (!isset(self::TYPE_LABELS[$type])) {
                $errors[] = 'Tipo de log inválido.';
            }
        } elseif ($start !== '' || $end !== '' || isset($_GET['filter'])) {
            $errors[] = 'Selecione o tipo de log para visualizar.';
        }

        $dateRange = $this->validateDateRange($start, $end);
        if ($dateRange['error'] !== null) {
            $errors[] = $dateRange['error'];
        }

        if ($errors === [] && $type !== '' && $dateRange['start'] !== null && $dateRange['end'] !== null) {
            $report = $this->resolveReport($type, $dateRange['start'], $dateRange['end'], $page);
            $report['type'] = $type;
            $report['title'] = self::TYPE_LABELS[$type];
        }

        return $this->render('admin/logs', [
            'pageTitle' => 'Relatório de Logs',
            'authUser' => $user,
            'adminSection' => 'logs',
            'errors' => $errors,
            'filters' => $filters,
            'report' => $report,
            'typeOptions' => self::TYPE_LABELS,
        ]);
    }

    private function resolveReport(string $type, string $startDate, string $endDate, int $page): array
    {
        return match ($type) {
            'application' => $this->logService->getApplicationLogs($startDate, $endDate, $page),
            'pdo' => $this->logService->getPdoLogs($startDate, $endDate, $page),
            'audit' => $this->logService->getAuditLogs($startDate, $endDate, $page),
            default => [
                'items' => [],
                'pagination' => [
                    'page' => 1,
                    'pages' => 1,
                    'per_page' => 1,
                    'total' => 0,
                ],
            ],
        };
    }

    /**
     * @return array{start: ?string, end: ?string, error: ?string}
     */
    private function validateDateRange(string $start, string $end): array
    {
        if ($start === '' && $end === '') {
            return ['start' => null, 'end' => null, 'error' => null];
        }

        if ($start === '' || $end === '') {
            return ['start' => null, 'end' => null, 'error' => 'Informe data inicial e final para aplicar o filtro.'];
        }

        $startDate = DateTimeImmutable::createFromFormat('Y-m-d', $start);
        $endDate = DateTimeImmutable::createFromFormat('Y-m-d', $end);
        if ($startDate === false || $endDate === false) {
            return ['start' => null, 'end' => null, 'error' => 'Datas inválidas. Utilize o formato AAAA-MM-DD.'];
        }

        if ($startDate > $endDate) {
            return ['start' => null, 'end' => null, 'error' => 'A data inicial deve ser anterior ou igual à data final.'];
        }

        $diff = $endDate->diff($startDate)->days ?? 0;
        if ($diff > self::MAX_RANGE_DAYS) {
            return ['start' => null, 'end' => null, 'error' => 'Selecione um intervalo de no máximo ' . self::MAX_RANGE_DAYS . ' dias.'];
        }

        return [
            'start' => $startDate->format('Y-m-d'),
            'end' => $endDate->format('Y-m-d'),
            'error' => null,
        ];
    }

    private function requireGestor(): array
    {
        $user = $this->authService->user();
        if ($user === null || ($user['role_slug'] ?? '') !== 'gestor') {
            throw new RuntimeException('Acesso restrito aos gestores.');
        }

        return $user;
    }
}
