<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Config\AppConfig;
use App\Config\Database;
use App\Repositories\NotificationOutboxRepository;
use Throwable;

final class StatusController extends Controller
{
    public function __construct(
        private readonly NotificationOutboxRepository $notifications
    ) {
    }

    public function index(): string
    {
        [$dbOk, $dbMessage] = $this->checkDatabase();
        $logDir = BASE_PATH . '/storage/logs';
        $uploadsDir = BASE_PATH . '/storage/uploads';
        $pendingNotifications = $this->notifications->pendingCount();

        $checks = [
            [
                'label' => 'Banco de Dados',
                'status' => $dbOk ? 'ok' : 'error',
                'details' => $dbMessage,
            ],
            [
                'label' => 'Fila de notificações',
                'status' => $pendingNotifications < 50 ? 'ok' : 'warning',
                'details' => $pendingNotifications . ' pendentes',
            ],
            [
                'label' => 'Logs graváveis',
                'status' => is_writable($logDir) ? 'ok' : 'error',
                'details' => $logDir,
            ],
            [
                'label' => 'Uploads graváveis',
                'status' => is_dir($uploadsDir) && is_writable($uploadsDir) ? 'ok' : 'warning',
                'details' => $uploadsDir,
            ],
        ];

        return $this->render('status/index', [
            'checks' => $checks,
            'meta' => [
                'time' => date('Y-m-d H:i:s'),
                'timezone' => AppConfig::timezone(),
                'php' => PHP_VERSION,
                'debug' => AppConfig::isDebug(),
                'cron_command' => 'php public/cron_send_notifications.php',
            ],
        ]);
    }

    /**
     * @return array{0: bool, 1: string}
     */
    private function checkDatabase(): array
    {
        try {
            Database::connection()->query('SELECT 1');
            return [true, 'Conectado'];
        } catch (Throwable $exception) {
            return [false, $exception->getMessage()];
        }
    }
}
