<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Repositories\LookupRepository;
use App\Services\AuthService;
use App\Services\TicketService;
use RuntimeException;

final class TicketController extends Controller
{
    public function __construct(
        private readonly AuthService $authService,
        private readonly TicketService $ticketService,
        private readonly LookupRepository $lookups
    ) {
    }

    public function index(): string
    {
        $user = $this->requireUser();
        $filters = $this->filtersFromRequest();
        $tickets = $this->ticketService->visibleTickets($user, $filters, 100);

        foreach ($tickets as &$ticket) {
            $ticket['sla'] = $this->ticketService->slaMetrics($ticket);
        }
        unset($ticket);

        return $this->render('tickets/index', [
            'tickets' => $tickets,
            'filters' => $filters,
            'options' => $this->lookupOptions(),
            'pageTitle' => 'Chamados',
            'flash' => $this->pullFlash('tickets_flash', []),
        ]);
    }

    public function myQueue(): string
    {
        $user = $this->requireUser();
        $role = $user['role_slug'] ?? '';
        if (!in_array($role, ['atendente-ti', 'atendente-noc'], true)) {
            return $this->render('tickets/not-found', ['message' => 'Acesso restrito aos atendentes.']);
        }

        $queueId = (int) ($user['queue_id'] ?? 0);
        if ($queueId === 0) {
            return $this->render('tickets/not-found', ['message' => 'Nenhuma fila vinculada ao seu perfil.']);
        }

        $scope = $_GET['scope'] ?? 'todos';
        if (!in_array($scope, ['todos', 'meus', 'livres'], true)) {
            $scope = 'todos';
        }
        $filters = [];

        if ($scope === 'meus') {
            $filters['assignee_id'] = (int) $user['id'];
        } elseif ($scope === 'livres') {
            $filters['unassigned'] = 1;
        }

        $tickets = $this->ticketService->visibleTickets($user, $filters, 100);
        foreach ($tickets as &$ticket) {
            $ticket['sla'] = $this->ticketService->slaMetrics($ticket);
        }
        unset($ticket);

        return $this->render('tickets/my-queue', [
            'tickets' => $tickets,
            'queueName' => $this->queueName($queueId),
            'summary' => $this->ticketService->queueSummary($user),
            'scope' => $scope,
            'pageTitle' => 'Minha fila',
        ]);
    }

    public function show(): string
    {
        $user = $this->requireUser();
        $ticketId = (int) ($_GET['id'] ?? 0);
        if ($ticketId <= 0) {
            return $this->render('tickets/not-found', ['message' => 'Ticket inválido']);
        }

        $ticket = $this->ticketService->ticketForUser($ticketId, $user);
        if ($ticket === null) {
            return $this->render('tickets/not-found', ['message' => 'Ticket não encontrado ou sem permissão.']);
        }

        $timeline = $this->ticketService->timeline($ticketId, $user);
        $dynamicFields = $this->ticketService->ticketFieldDisplay($ticketId);
        $sla = $this->ticketService->slaMetrics($ticket);
        $commentErrors = $this->pullFlash('comment_errors', []);
        $commentOld = $this->pullFlash('comment_old', []);
        $emergencyErrors = $this->pullFlash('emergency_errors', []);
        $emergencyOld = $this->pullFlash('emergency_old', []);
        $isEmergency = !empty($ticket['is_emergency']);
        $emergencyLogs = $isEmergency ? $this->ticketService->emergencyActionLogs($ticketId, $user) : [];

        return $this->render('tickets/show', [
            'ticket' => $ticket,
            'timeline' => $timeline,
            'sla' => $sla,
            'dynamicFields' => $dynamicFields,
            'commentErrors' => $commentErrors,
            'commentOld' => $commentOld,
            'emergencyErrors' => $emergencyErrors,
            'emergencyOld' => $emergencyOld,
            'emergencyActions' => $emergencyLogs,
            'isEmergency' => $isEmergency,
            'pageTitle' => 'Chamado ' . $ticket['reference'],
            'flash' => $this->pullFlash('tickets_flash', []),
        ]);
    }

    public function create(): string
    {
        $user = $this->requireUser();
        $old = $this->pullFlash('ticket_old', []);
        $errors = $this->pullFlash('ticket_errors', []);
        $oldFields = [];

        if (isset($old['fields']) && is_array($old['fields'])) {
            $oldFields = $old['fields'];
            unset($old['fields']);
        }

        $options = $this->lookupOptions();
        $defaults = [
            'client_type' => $this->defaultClientType($user, $options['customer_types'] ?? []),
            'main_category' => $this->defaultMainCategory($user, $options['main_categories'] ?? []),
            'requester_id' => $user['id'],
        ];

        return $this->render('tickets/form', [
            'mode' => 'create',
            'ticket' => array_merge($defaults, $old),
            'options' => $options,
            'errors' => $errors,
            'fieldValues' => $oldFields,
            'dynamicFieldCatalog' => $this->ticketService->fieldCatalog(),
            'pageTitle' => 'Novo Chamado',
            'currentUser' => $user,
        ]);
    }

    public function store(): void
    {
        $user = $this->requireUser();

        try {
            csrf_validate();
            $ticketId = $this->ticketService->createTicket($_POST, $user);
            $this->flash('tickets_flash', ['success' => 'Chamado criado com sucesso!']);
            header('Location: ' . app_url('tickets/show?id=' . $ticketId));
            exit;
        } catch (RuntimeException $exception) {
            $this->flash('ticket_errors', [$exception->getMessage()]);
            $this->flash('ticket_old', $_POST);
            header('Location: ' . app_url('tickets/create'));
            exit;
        }
    }

    public function edit(): string
    {
        $user = $this->requireUser();
        $ticketId = (int) ($_GET['id'] ?? 0);
        if ($ticketId <= 0) {
            return $this->render('tickets/not-found', ['message' => 'Ticket inválido']);
        }

        $ticket = $this->ticketService->ticketForUser($ticketId, $user);
        if ($ticket === null || !\can('update_ticket')) {
            return $this->render('tickets/not-found', ['message' => 'Ticket não encontrado ou sem permissão.']);
        }

        $old = $this->pullFlash('ticket_old', []);
        $errors = $this->pullFlash('ticket_errors', []);
        $oldFields = [];

        if (isset($old['fields']) && is_array($old['fields'])) {
            $oldFields = $old['fields'];
            unset($old['fields']);
        }

        $fieldValues = $oldFields !== [] ? $oldFields : $this->ticketService->ticketFieldValues((int) $ticket['id']);

        return $this->render('tickets/form', [
            'mode' => 'edit',
            'ticket' => array_merge($ticket, $old),
            'options' => $this->lookupOptions(),
            'errors' => $errors,
            'fieldValues' => $fieldValues,
            'dynamicFieldCatalog' => $this->ticketService->fieldCatalog(),
            'pageTitle' => 'Editar ' . $ticket['reference'],
            'currentUser' => $user,
        ]);
    }

    public function update(): void
    {
        $user = $this->requireUser();
        $ticketId = (int) ($_GET['id'] ?? 0);

        if ($ticketId <= 0) {
            $this->flash('ticket_errors', ['Ticket inválido.']);
            header('Location: ' . app_url('tickets'));
            exit;
        }

        try {
            csrf_validate();
            $this->ticketService->updateTicket($ticketId, $_POST, $user);
            $this->flash('tickets_flash', ['success' => 'Chamado atualizado.']);
            header('Location: ' . app_url('tickets/show?id=' . $ticketId));
            exit;
        } catch (RuntimeException $exception) {
            $this->flash('ticket_errors', [$exception->getMessage()]);
            $this->flash('ticket_old', $_POST);
            header('Location: ' . app_url('tickets/edit?id=' . $ticketId));
            exit;
        }
    }

    public function destroy(): void
    {
        $user = $this->requireUser();
        $ticketId = (int) ($_POST['id'] ?? 0);

        if ($ticketId <= 0) {
            $this->flash('tickets_flash', ['error' => 'Ticket inválido.']);
            header('Location: ' . app_url('tickets'));
            exit;
        }

        try {
            csrf_validate();
            $this->ticketService->deleteTicket($ticketId, $user);
            $this->flash('tickets_flash', ['success' => 'Chamado removido.']);
        } catch (RuntimeException $exception) {
            $this->flash('tickets_flash', ['error' => $exception->getMessage()]);
        }

        header('Location: ' . app_url('tickets'));
        exit;
    }

    public function comment(): void
    {
        $user = $this->requireUser();
        $ticketId = (int) ($_POST['ticket_id'] ?? 0);

        if ($ticketId <= 0) {
            $this->flash('tickets_flash', ['error' => 'Ticket inválido.']);
            header('Location: ' . app_url('tickets'));
            exit;
        }

        try {
            csrf_validate();
            $this->ticketService->addComment($ticketId, $_POST, $_FILES, $user);
            $this->flash('tickets_flash', ['success' => 'Comentário registrado.']);
        } catch (RuntimeException $exception) {
            $this->flash('comment_errors', [$exception->getMessage()]);
            $this->flash('comment_old', [
                'content' => trim((string) ($_POST['content'] ?? '')),
                'is_internal' => isset($_POST['is_internal']) ? 1 : 0,
            ]);
        }

        header('Location: ' . app_url('tickets/show?id=' . $ticketId));
        exit;
    }

    public function logEmergencyAction(): void
    {
        $user = $this->requireUser();
        $ticketId = (int) ($_POST['ticket_id'] ?? 0);

        if ($ticketId <= 0) {
            $this->flash('tickets_flash', ['error' => 'Ticket inválido.']);
            header('Location: ' . app_url('tickets'));
            exit;
        }

        try {
            csrf_validate();
            $this->ticketService->logEmergencyAction($ticketId, $_POST, $user);
            $this->flash('tickets_flash', ['success' => 'Registro de emergência adicionado.']);
        } catch (RuntimeException $exception) {
            $this->flash('emergency_errors', [$exception->getMessage()]);
            $this->flash('emergency_old', $_POST);
        }

        header('Location: ' . app_url('tickets/show?id=' . $ticketId));
        exit;
    }

    private function requireUser(): array
    {
        $user = $this->authService->user();
        if ($user === null) {
            throw new RuntimeException('Usuário não autenticado.');
        }

        return $user;
    }

    /**
     * @return array<string, mixed>
     */
    private function filtersFromRequest(): array
    {
        return [
            'status_id' => $_GET['status_id'] ?? '',
            'priority_id' => $_GET['priority_id'] ?? '',
            'category_id' => $_GET['category_id'] ?? '',
            'queue_id' => $_GET['queue_id'] ?? '',
            'main_category' => $_GET['main_category'] ?? '',
            'from_date' => $_GET['from_date'] ?? '',
            'to_date' => $_GET['to_date'] ?? '',
        ];
    }

    /**
     * @return array<string, array<int, array<string, mixed>>>
     */
    private function lookupOptions(): array
    {
        return [
            'statuses' => $this->lookups->statuses(),
            'priorities' => $this->lookups->priorities(),
            'queues' => $this->lookups->queues(),
            'categories' => $this->lookups->categories(),
            'users' => $this->lookups->activeUsers(),
            'customer_types' => $this->lookups->customerTypes(),
            'main_categories' => $this->lookups->mainCategories(),
        ];
    }

    /**
     * @param array<int, array<string, mixed>> $customerTypes
     */
    private function defaultClientType(array $user, array $customerTypes): string
    {
        $preferred = $user['role_slug'] === 'externo' ? 'externo' : 'interno';
        foreach ($customerTypes as $type) {
            if (strcasecmp((string) ($type['slug'] ?? ''), $preferred) === 0) {
                return (string) $type['slug'];
            }
        }

        return (string) ($customerTypes[0]['slug'] ?? $preferred);
    }

    /**
     * @param array<int, array<string, mixed>> $mainCategories
     */
    private function defaultMainCategory(array $user, array $mainCategories): string
    {
        $preferred = 'ti';
        foreach ($mainCategories as $category) {
            if (strcasecmp((string) ($category['slug'] ?? ''), $preferred) === 0) {
                return (string) $category['slug'];
            }
        }

        return (string) ($mainCategories[0]['slug'] ?? $preferred);
    }

    private function queueName(int $queueId): string
    {
        $queues = $this->lookups->queues();
        foreach ($queues as $queue) {
            if ((int) $queue['id'] === $queueId) {
                return (string) $queue['name'];
            }
        }

        return 'Fila #' . $queueId;
    }

    private function flash(string $key, array $value): void
    {
        $_SESSION[$key] = $value;
    }

    private function pullFlash(string $key, array $default = []): array
    {
        if (!isset($_SESSION[$key])) {
            return $default;
        }

        $value = $_SESSION[$key];
        unset($_SESSION[$key]);

        return is_array($value) ? $value : $default;
    }
}
