<?php
declare(strict_types=1);

/**
 * Returns the authenticated user array stored in the session.
 */
function auth_user(): ?array
{
    return $_SESSION['auth_user'] ?? null;
}

/**
 * Checks whether the authenticated user belongs to any of the given roles.
 */
function role(string ...$roles): bool
{
    $user = auth_user();
    if ($user === null) {
        return false;
    }

    if ($roles === []) {
        return true;
    }

    return in_array($user['role_slug'] ?? '', $roles, true);
}

/**
 * Generic permission helper used across controllers and services.
 *
 * @param array<string, mixed> $context
 */
function can(string $ability, array $context = []): bool
{
    $user = auth_user();
    if ($user === null) {
        return false;
    }

    $role = $user['role_slug'] ?? '';

    return match ($ability) {
        'view_dashboard',
        'view_tickets' => in_array($role, ['gestor', 'atendente-ti', 'atendente-noc', 'interno', 'externo'], true),
        'view_ticket' => canViewTicket($role, $user, $context),
        'manage_catalog',
        'manage_sla',
        'view_all_tickets' => $role === 'gestor',
        'create_ticket' => in_array($role, ['gestor', 'atendente-ti', 'atendente-noc', 'interno', 'externo'], true),
        'update_ticket' => in_array($role, ['gestor', 'atendente-ti', 'atendente-noc'], true),
        'delete_ticket' => $role === 'gestor',
        default => false,
    };
}

/**
 * @param array<string, mixed> $user
 * @param array<string, mixed> $context
 */
function canViewTicket(string $role, array $user, array $context): bool
{
    if ($role === 'gestor') {
        return true;
    }

    if (in_array($role, ['atendente-ti', 'atendente-noc'], true)) {
        $queueId = (int) ($user['queue_id'] ?? 0);
        if ($queueId === 0) {
            return false;
        }

        $ticketQueue = isset($context['queue_id']) ? (int) $context['queue_id'] : null;
        return $ticketQueue === null || $ticketQueue === $queueId;
    }

    if (in_array($role, ['interno', 'externo'], true)) {
        $ticketRequester = isset($context['requester_id']) ? (int) $context['requester_id'] : null;
        return $ticketRequester === null || $ticketRequester === (int) $user['id'];
    }

    return false;
}
