<?php
declare(strict_types=1);

if (!function_exists('app_log')) {
    require_once __DIR__ . '/logger.php';
}

function csrf_token(): string
{
    if (!isset($_SESSION['_csrf_token']) || !is_string($_SESSION['_csrf_token'])) {
        $_SESSION['_csrf_token'] = bin2hex(random_bytes(32));
    }

    return $_SESSION['_csrf_token'];
}

function csrf_input(): string
{
    $token = htmlspecialchars(csrf_token(), ENT_QUOTES, 'UTF-8');
    return '<input type="hidden" name="_token" value="' . $token . '">';
}

/**
 * @throws RuntimeException
 */
function csrf_validate(): void
{
    if (session_status() !== PHP_SESSION_ACTIVE) {
        csrf_log_failure('missing_session', ['uri' => $_SERVER['REQUEST_URI'] ?? 'cli']);
        throw new RuntimeException('Sessão expirada. Recarregue a página e tente novamente.');
    }

    $token = $_POST['_token'] ?? '';
    if (!is_string($token) || $token === '') {
        csrf_log_failure('missing_token', ['uri' => $_SERVER['REQUEST_URI'] ?? 'cli']);
        throw new RuntimeException('Sessão expirada. Recarregue a página e tente novamente.');
    }

    $sessionToken = $_SESSION['_csrf_token'] ?? null;
    if (!is_string($sessionToken) || $sessionToken === '') {
        csrf_log_failure('missing_session_token', ['uri' => $_SERVER['REQUEST_URI'] ?? 'cli']);
        throw new RuntimeException('Sessão expirada. Recarregue a página e tente novamente.');
    }

    if (!hash_equals($sessionToken, $token)) {
        csrf_log_failure('token_mismatch', [
            'uri' => $_SERVER['REQUEST_URI'] ?? 'cli',
            'expected_prefix' => substr($sessionToken, 0, 8),
            'provided_prefix' => substr($token, 0, 8),
        ]);
        throw new RuntimeException('Token de segurança inválido. Recarregue a página e tente novamente.');
    }
}

function csrf_log_failure(string $reason, array $context = []): void
{
    if (!function_exists('app_log')) {
        error_log('[CSRF] ' . $reason . ' ' . json_encode($context));
        return;
    }

    $payload = array_merge(['reason' => $reason], $context);
    app_log('WARNING', 'CSRF validation falhou', $payload);
}
