<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class AdminAuditRepository
{
    /**
     * @param array<string, mixed> $payload
     */
    public function log(?int $userId, string $entity, string $action, ?int $entityId, array $payload = []): void
    {
        $sql = <<<SQL
            INSERT INTO admin_audit_logs (user_id, entity, action, entity_id, payload)
            VALUES (:user_id, :entity, :action, :entity_id, :payload)
        SQL;

        Database::run($sql, [
            'user_id' => $userId,
            'entity' => $entity,
            'action' => $action,
            'entity_id' => $entityId,
            'payload' => $payload === [] ? null : json_encode($payload, JSON_UNESCAPED_UNICODE),
        ]);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function latest(int $limit = 25): array
    {
        $sql = <<<SQL
            SELECT
                log.id,
                log.entity,
                log.action,
                log.entity_id,
                log.payload,
                log.created_at,
                u.name AS user_name
            FROM admin_audit_logs log
            LEFT JOIN users u ON u.id = log.user_id
            ORDER BY log.created_at DESC, log.id DESC
            LIMIT :limit
        SQL;

        $statement = Database::connection()->prepare($sql);
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->execute();

        $rows = $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
        foreach ($rows as &$row) {
            $row['payload'] = $row['payload'] !== null ? json_decode((string) $row['payload'], true) : null;
        }
        unset($row);

        return $rows;
    }

    /**
     * @return array{rows: array<int, array<string, mixed>>, total: int}
     */
    public function search(?string $startDateTime, ?string $endDateTime, int $limit, int $offset): array
    {
        $conditions = [];
        $params = [];

        if ($startDateTime !== null) {
            $conditions[] = 'log.created_at >= :start';
            $params['start'] = $startDateTime;
        }

        if ($endDateTime !== null) {
            $conditions[] = 'log.created_at <= :end';
            $params['end'] = $endDateTime;
        }

        $where = $conditions === [] ? '' : 'WHERE ' . implode(' AND ', $conditions);

        $countSql = 'SELECT COUNT(*) FROM admin_audit_logs log ' . $where;
        $countStatement = Database::prepare($countSql);
        foreach ($params as $key => $value) {
            $countStatement->bindValue(':' . $key, $value);
        }
        $countStatement->execute();
        $total = (int) $countStatement->fetchColumn();

        $sql = <<<SQL
            SELECT
                log.id,
                log.entity,
                log.action,
                log.entity_id,
                log.payload,
                log.created_at,
                u.name AS user_name
            FROM admin_audit_logs log
            LEFT JOIN users u ON u.id = log.user_id
            $where
            ORDER BY log.created_at DESC, log.id DESC
            LIMIT :limit OFFSET :offset
        SQL;

        $statement = Database::connection()->prepare($sql);
        foreach ($params as $key => $value) {
            $statement->bindValue(':' . $key, $value);
        }
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->bindValue(':offset', $offset, PDO::PARAM_INT);
        $statement->execute();

        $rows = $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
        foreach ($rows as &$row) {
            $row['payload'] = $row['payload'] !== null ? json_decode((string) $row['payload'], true) : null;
        }
        unset($row);

        return [
            'rows' => $rows,
            'total' => $total,
        ];
    }
}
