<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;
use RuntimeException;

final class CategoryRepository
{
    /**
     * @var array<int, array<string, mixed>>
     */
    private array $cache = [];

    /**
     * @var array<string, bool>
     */
    private array $columnPresence = [];

    public function find(int $categoryId): ?array
    {
        if (isset($this->cache[$categoryId])) {
            return $this->cache[$categoryId];
        }

        $select = $this->baseSelectColumns();
        $sql = <<<SQL
            SELECT {$select}
            FROM categories c
            WHERE c.id = :id
            LIMIT 1
        SQL;

        $statement = Database::run($sql, ['id' => $categoryId]);
        $row = $statement->fetch(PDO::FETCH_ASSOC);
        if ($row === false) {
            return null;
        }

        $row['id'] = (int) $row['id'];
        $row['parent_id'] = $row['parent_id'] !== null ? (int) $row['parent_id'] : null;
        $row['queue_id'] = $row['queue_id'] !== null ? (int) $row['queue_id'] : null;
        $row['sort_order'] = (int) ($row['sort_order'] ?? 0);
        $row['is_active'] = (int) ($row['is_active'] ?? 1);
        $this->cache[$categoryId] = $row;

        return $row;
    }

    public function findById(int $id): ?array
    {
        return $this->find($id);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function all(): array
    {
        $select = $this->baseSelectColumns();
        $order = $this->orderClause();
        $sql = <<<SQL
            SELECT
                {$select},
                parent.name AS parent_name
            FROM categories c
            LEFT JOIN categories parent ON parent.id = c.parent_id
            {$order}
        SQL;

        $statement = Database::run($sql);
        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function parentOptions(?int $excludeId = null, ?int $includeId = null): array
    {
        $clauses = [];
        $params = [];

        if ($this->hasColumn('is_active')) {
            if ($includeId !== null) {
                $clauses[] = '(is_active = 1 OR id = :include_id)';
                $params['include_id'] = $includeId;
            } else {
                $clauses[] = 'is_active = 1';
            }
        }

        if ($excludeId !== null) {
            $clauses[] = 'id <> :exclude_id';
            $params['exclude_id'] = $excludeId;
        }

        $where = $clauses === [] ? '' : 'WHERE ' . implode(' AND ', $clauses);
        $sql = 'SELECT id, name FROM categories ' . $where . ' ORDER BY name ASC';

        $statement = Database::run($sql, $params);
        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function mainCategories(): array
    {
        $conditions = ['parent_id IS NULL'];
        if ($this->hasColumn('is_active')) {
            $conditions[] = 'is_active = 1';
        }
        $where = 'WHERE ' . implode(' AND ', $conditions);
        $sql = <<<SQL
            SELECT id, name, slug, description, queue_id, sort_order
            FROM categories
            {$where}
            ORDER BY sort_order ASC, name ASC
        SQL;

        $statement = Database::run($sql);
        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array{rows: array<int, array<string, mixed>>, total: int}
     */
    public function paginate(?string $search, ?string $status, int $limit, int $offset): array
    {
        $hasStatus = $this->hasColumn('is_active');
        $conditions = ['1=1'];
        $params = [];

        if ($search !== null && $search !== '') {
            $conditions[] = '(c.name LIKE :search OR c.slug LIKE :search)';
            $params['search'] = '%' . $search . '%';
        }

        if ($hasStatus) {
            if ($status === 'active') {
                $conditions[] = 'c.is_active = 1';
            } elseif ($status === 'inactive') {
                $conditions[] = 'c.is_active = 0';
            }
        } else {
            $status = null;
        }

        $where = 'WHERE ' . implode(' AND ', $conditions);

        $countSql = 'SELECT COUNT(*) FROM categories c ' . $where;
        $countStmt = Database::prepare($countSql);
        foreach ($params as $key => $value) {
            $countStmt->bindValue(':' . $key, $value);
        }
        $countStmt->execute();
        $total = (int) $countStmt->fetchColumn();

        $select = $this->baseSelectColumns();
        $orderParts = ['sort_order ASC', 'c.name ASC'];
        $order = 'ORDER BY ' . implode(', ', $orderParts);

        $sql = <<<SQL
            SELECT
                {$select},
                parent.name AS parent_name,
                q.name AS queue_name
            FROM categories c
            LEFT JOIN categories parent ON parent.id = c.parent_id
            LEFT JOIN queues q ON q.id = c.queue_id
            {$where}
            {$order}
            LIMIT :limit OFFSET :offset
        SQL;

        $statement = Database::connection()->prepare($sql);
        foreach ($params as $key => $value) {
            $statement->bindValue(':' . $key, $value);
        }
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->bindValue(':offset', $offset, PDO::PARAM_INT);
        $statement->execute();

        return [
            'rows' => $statement->fetchAll(PDO::FETCH_ASSOC) ?: [],
            'total' => $total,
        ];
    }

    /**
     * @param array<string, mixed> $data
     */
    public function create(array $data): int
    {
        $payload = $this->filterColumns($data);
        $columns = array_keys($payload);
        $placeholders = array_map(static fn(string $column): string => ':' . $column, $columns);

        $sql = sprintf(
            'INSERT INTO categories (%s) VALUES (%s)',
            implode(', ', $columns),
            implode(', ', $placeholders)
        );

        Database::run($sql, $payload);
        $categoryId = (int) Database::connection()->lastInsertId();
        $this->cache = [];

        return $categoryId;
    }

    /**
     * @param array<string, mixed> $data
     */
    public function update(int $id, array $data): bool
    {
        $payload = $this->filterColumns($data);
        if ($payload === []) {
            return true;
        }

        $fields = [];
        foreach ($payload as $column => $_value) {
            $fields[] = sprintf('%s = :%s', $column, $column);
        }
        $fields[] = 'updated_at = CURRENT_TIMESTAMP';

        $sql = sprintf(
            'UPDATE categories SET %s WHERE id = :id',
            implode(', ', $fields)
        );
        $payload['id'] = $id;

        $statement = Database::run($sql, $payload);
        $this->cache = [];

        return $statement->rowCount() > 0;
    }

    public function delete(int $id): bool
    {
        $statement = Database::run('DELETE FROM categories WHERE id = :id', ['id' => $id]);
        $this->cache = [];

        return $statement->rowCount() > 0;
    }

    public function toggleStatus(int $id, int $status): bool
    {
        if (!$this->hasColumn('is_active')) {
            throw new RuntimeException('A coluna is_active não está disponível na tabela categories.');
        }

        $statement = Database::run(
            'UPDATE categories SET is_active = :is_active, updated_at = NOW() WHERE id = :id',
            [
                'id' => $id,
                'is_active' => $status,
            ]
        );
        $this->cache = [];

        return $statement->rowCount() > 0;
    }

    public function supportsStatus(): bool
    {
        return $this->hasColumn('is_active');
    }

    public function slugExists(string $slug, ?int $ignoreId = null): bool
    {
        $sql = 'SELECT 1 FROM categories WHERE slug = :slug';
        $params = ['slug' => $slug];

        if ($ignoreId !== null) {
            $sql .= ' AND id <> :ignore_id';
            $params['ignore_id'] = $ignoreId;
        }

        $sql .= ' LIMIT 1';

        $statement = Database::run($sql, $params);
        return $statement->fetchColumn() !== false;
    }

    public function nameExists(string $name, ?int $ignoreId = null): bool
    {
        $sql = 'SELECT 1 FROM categories WHERE name = :name';
        $params = ['name' => $name];

        if ($ignoreId !== null) {
            $sql .= ' AND id <> :ignore_id';
            $params['ignore_id'] = $ignoreId;
        }

        $sql .= ' LIMIT 1';

        $statement = Database::run($sql, $params);
        return $statement->fetchColumn() !== false;
    }

    public function isInUse(int $id): bool
    {
        $checks = [
            'SELECT 1 FROM categories WHERE parent_id = :id LIMIT 1',
            'SELECT 1 FROM tickets WHERE category_id = :id LIMIT 1',
            'SELECT 1 FROM ticket_fields WHERE category_id = :id LIMIT 1',
            'SELECT 1 FROM sla_policies WHERE category_id = :id LIMIT 1',
        ];

        foreach ($checks as $sql) {
            $statement = Database::run($sql, ['id' => $id]);
            if ($statement->fetchColumn() !== false) {
                return true;
            }
        }

        return false;
    }

    public function queueIdForCategory(int $categoryId): ?int
    {
        $category = $this->find($categoryId);
        if ($category === null) {
            return null;
        }

        if (!empty($category['queue_id'])) {
            return (int) $category['queue_id'];
        }

        $parentId = $category['parent_id'] ?? null;
        if ($parentId === null) {
            return null;
        }

        return $this->queueIdForCategory((int) $parentId);
    }

    private function baseSelectColumns(): string
    {
        $columns = [
            'c.id',
            'c.parent_id',
            'c.name',
            'c.slug',
            'c.description',
            'c.queue_id',
        ];

        if ($this->hasColumn('sort_order')) {
            $columns[] = 'c.sort_order AS sort_order';
        } else {
            $columns[] = '0 AS sort_order';
        }

        if ($this->hasColumn('is_active')) {
            $columns[] = 'c.is_active';
        } else {
            $columns[] = '1 AS is_active';
        }

        return implode(', ', $columns);
    }

    private function orderClause(): string
    {
        $parts = [];
        $parts[] = 'sort_order ASC';
        $parts[] = 'c.name ASC';

        return 'ORDER BY ' . implode(', ', $parts);
    }

    /**
     * @param array<string, mixed> $data
     * @return array<string, mixed>
     */
    private function filterColumns(array $data): array
    {
        $payload = [
            'parent_id' => $data['parent_id'] ?? null,
            'name' => $data['name'] ?? null,
            'slug' => $data['slug'] ?? null,
            'description' => $data['description'] ?? null,
            'queue_id' => $data['queue_id'] ?? null,
        ];

        if ($this->hasColumn('sort_order') && array_key_exists('sort_order', $data)) {
            $payload['sort_order'] = $data['sort_order'];
        }
        if ($this->hasColumn('is_active') && array_key_exists('is_active', $data)) {
            $payload['is_active'] = $data['is_active'];
        }

        return $payload;
    }

    private function hasColumn(string $column): bool
    {
        if (!array_key_exists($column, $this->columnPresence)) {
            $statement = Database::run(
                'SHOW COLUMNS FROM categories LIKE :column',
                ['column' => $column]
            );
            $this->columnPresence[$column] = $statement->fetch(PDO::FETCH_ASSOC) !== false;
        }

        return $this->columnPresence[$column];
    }
}
