<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class CommentRepository
{
    /**
     * @return array<int, array<string, mixed>>
     */
    public function listByTicket(int $ticketId): array
    {
        $sql = <<<SQL
            SELECT
                c.id,
                c.ticket_id,
                c.author_id,
                author.name AS author_name,
                c.content,
                c.is_internal,
                c.created_at
            FROM ticket_comments c
            LEFT JOIN users author ON author.id = c.author_id
            WHERE c.ticket_id = :ticket_id
            ORDER BY c.created_at ASC
        SQL;

        $statement = Database::run($sql, ['ticket_id' => $ticketId]);

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    public function create(int $ticketId, ?int $authorId, string $content, bool $isInternal): int
    {
        $sql = <<<SQL
            INSERT INTO ticket_comments (ticket_id, author_id, content, is_internal)
            VALUES (:ticket_id, :author_id, :content, :is_internal)
        SQL;

        Database::run($sql, [
            'ticket_id' => $ticketId,
            'author_id' => $authorId,
            'content' => $content,
            'is_internal' => $isInternal ? 1 : 0,
        ]);

        return (int) Database::connection()->lastInsertId();
    }
}
