<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class CustomerTypeRepository
{
    /**
     * @return array<int, array<string, mixed>>
     */
    public function all(): array
    {
        $sql = 'SELECT id, slug, name, description, sort_order, is_active FROM customer_types ORDER BY sort_order ASC, name ASC';
        $statement = Database::run($sql);

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function active(): array
    {
        $sql = 'SELECT id, slug, name FROM customer_types WHERE is_active = 1 ORDER BY sort_order ASC, name ASC';
        $statement = Database::run($sql);

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    public function findById(int $id): ?array
    {
        $statement = Database::run('SELECT id, slug, name, description, sort_order, is_active FROM customer_types WHERE id = :id LIMIT 1', ['id' => $id]);
        $row = $statement->fetch(PDO::FETCH_ASSOC);

        if ($row === false) {
            return null;
        }

        $row['id'] = (int) $row['id'];
        $row['sort_order'] = (int) ($row['sort_order'] ?? 0);
        $row['is_active'] = (int) ($row['is_active'] ?? 1);

        return $row;
    }

    /**
     * @return array{rows: array<int, array<string, mixed>>, total: int}
     */
    public function paginate(?string $search, ?string $status, int $limit, int $offset): array
    {
        $conditions = ['1=1'];
        $params = [];

        if ($search !== null && $search !== '') {
            $conditions[] = '(slug LIKE :search OR name LIKE :search)';
            $params['search'] = '%' . $search . '%';
        }

        if ($status === 'active') {
            $conditions[] = 'is_active = 1';
        } elseif ($status === 'inactive') {
            $conditions[] = 'is_active = 0';
        }

        $where = 'WHERE ' . implode(' AND ', $conditions);

        $countStmt = Database::prepare('SELECT COUNT(*) FROM customer_types ' . $where);
        foreach ($params as $name => $value) {
            $countStmt->bindValue(':' . $name, $value);
        }
        $countStmt->execute();
        $total = (int) $countStmt->fetchColumn();

        $sql = <<<SQL
            SELECT id, slug, name, description, sort_order, is_active
            FROM customer_types
            {$where}
            ORDER BY sort_order ASC, name ASC
            LIMIT :limit OFFSET :offset
        SQL;
        $statement = Database::connection()->prepare($sql);
        foreach ($params as $name => $value) {
            $statement->bindValue(':' . $name, $value);
        }
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->bindValue(':offset', $offset, PDO::PARAM_INT);
        $statement->execute();

        return [
            'rows' => $statement->fetchAll(PDO::FETCH_ASSOC) ?: [],
            'total' => $total,
        ];
    }

    /**
     * @param array<string, mixed> $data
     */
    public function create(array $data): int
    {
        $columns = array_keys($data);
        $placeholders = array_map(static fn(string $column): string => ':' . $column, $columns);
        $sql = sprintf(
            'INSERT INTO customer_types (%s) VALUES (%s)',
            implode(', ', $columns),
            implode(', ', $placeholders)
        );

        Database::run($sql, $data);

        return (int) Database::connection()->lastInsertId();
    }

    /**
     * @param array<string, mixed> $data
     */
    public function update(int $id, array $data): bool
    {
        if ($data === []) {
            return true;
        }

        $set = [];
        foreach ($data as $column => $_) {
            $set[] = sprintf('%s = :%s', $column, $column);
        }
        $sql = sprintf('UPDATE customer_types SET %s WHERE id = :id', implode(', ', $set));
        $data['id'] = $id;
        $statement = Database::run($sql, $data);

        return $statement->rowCount() > 0;
    }

    public function delete(int $id): bool
    {
        $statement = Database::run('DELETE FROM customer_types WHERE id = :id', ['id' => $id]);

        return $statement->rowCount() > 0;
    }

    public function toggleStatus(int $id, int $status): bool
    {
        $statement = Database::run(
            'UPDATE customer_types SET is_active = :status WHERE id = :id',
            ['status' => $status, 'id' => $id]
        );

        return $statement->rowCount() > 0;
    }

    public function nameExists(string $name, ?int $ignoreId = null): bool
    {
        $sql = 'SELECT COUNT(*) FROM customer_types WHERE name = :name';
        $params = ['name' => $name];
        if ($ignoreId !== null) {
            $sql .= ' AND id <> :id';
            $params['id'] = $ignoreId;
        }

        $statement = Database::run($sql, $params);

        return (int) $statement->fetchColumn() > 0;
    }

    public function slugExists(string $slug, ?int $ignoreId = null): bool
    {
        $sql = 'SELECT COUNT(*) FROM customer_types WHERE slug = :slug';
        $params = ['slug' => $slug];
        if ($ignoreId !== null) {
            $sql .= ' AND id <> :id';
            $params['id'] = $ignoreId;
        }

        $statement = Database::run($sql, $params);

        return (int) $statement->fetchColumn() > 0;
    }

    public function isInUse(int $id): bool
    {
        $type = $this->findById($id);
        if ($type === null) {
            return false;
        }

        $slug = (string) $type['slug'];
        $ticketCount = Database::run(
            'SELECT COUNT(*) FROM tickets WHERE client_type = :slug',
            ['slug' => $slug]
        )->fetchColumn();

        if ((int) $ticketCount > 0) {
            return true;
        }

        $slaCount = Database::run(
            'SELECT COUNT(*) FROM sla_policies WHERE client_type = :slug',
            ['slug' => $slug]
        )->fetchColumn();

        return (int) $slaCount > 0;
    }
}
