<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class SlaRepository
{
    /**
     * @return array{rows: array<int, array<string, mixed>>, total: int}
     */
    public function paginate(?string $search, ?string $status, int $limit, int $offset): array
    {
        $conditions = ['s.deleted_at IS NULL'];
        $params = [];

        if ($search !== null && $search !== '') {
            $conditions[] = 's.name LIKE :search';
            $params['search'] = '%' . $search . '%';
        }

        if ($status === 'active') {
            $conditions[] = 's.is_active = 1';
        } elseif ($status === 'inactive') {
            $conditions[] = 's.is_active = 0';
        }

        $where = 'WHERE ' . implode(' AND ', $conditions);

        $countSql = 'SELECT COUNT(*) FROM slas s ' . $where;
        $countStmt = Database::prepare($countSql);
        foreach ($params as $key => $value) {
            $countStmt->bindValue(':' . $key, $value);
        }
        $countStmt->execute();
        $total = (int) $countStmt->fetchColumn();

        $sql = <<<SQL
            SELECT
                s.*,
                p.name AS priority_name
            FROM slas s
            LEFT JOIN priorities p ON p.id = s.priority_id
            $where
            ORDER BY s.created_at DESC, s.id DESC
            LIMIT :limit OFFSET :offset
        SQL;

        $statement = Database::connection()->prepare($sql);
        foreach ($params as $key => $value) {
            $statement->bindValue(':' . $key, $value);
        }
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->bindValue(':offset', $offset, PDO::PARAM_INT);
        $statement->execute();

        return [
            'rows' => $statement->fetchAll(PDO::FETCH_ASSOC) ?: [],
            'total' => $total,
        ];
    }

    public function findById(int $id): ?array
    {
        $sql = <<<SQL
            SELECT s.*, p.name AS priority_name
            FROM slas s
            LEFT JOIN priorities p ON p.id = s.priority_id
            WHERE s.id = :id
              AND s.deleted_at IS NULL
            LIMIT 1
        SQL;

        $statement = Database::run($sql, ['id' => $id]);
        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? null : $row;
    }

    /**
     * @param array<string, mixed> $data
     */
    public function create(array $data): int
    {
        $sql = <<<SQL
            INSERT INTO slas (
                name,
                description,
                first_response_minutes,
                resolution_minutes,
                business_hours_only,
                escalation_enabled,
                escalation_minutes,
                priority_id,
                is_active
            ) VALUES (
                :name,
                :description,
                :first_response_minutes,
                :resolution_minutes,
                :business_hours_only,
                :escalation_enabled,
                :escalation_minutes,
                :priority_id,
                :is_active
            )
        SQL;

        Database::run($sql, $data);
        return (int) Database::connection()->lastInsertId();
    }

    /**
     * @param array<string, mixed> $data
     */
    public function update(int $id, array $data): bool
    {
        $fields = [];
        foreach ($data as $column => $_value) {
            $fields[] = sprintf('%s = :%s', $column, $column);
        }
        $fields[] = 'updated_at = CURRENT_TIMESTAMP';

        $sql = sprintf(
            'UPDATE slas SET %s WHERE id = :id AND deleted_at IS NULL',
            implode(', ', $fields)
        );
        $data['id'] = $id;

        $statement = Database::run($sql, $data);
        return $statement->rowCount() > 0;
    }

    public function softDelete(int $id): bool
    {
        $sql = 'UPDATE slas SET deleted_at = NOW(), updated_at = NOW() WHERE id = :id AND deleted_at IS NULL';
        $statement = Database::run($sql, ['id' => $id]);
        return $statement->rowCount() > 0;
    }

    public function isInUse(int $id): bool
    {
        // There are currently no foreign keys referencing slas.
        return false;
    }

    public function toggleStatus(int $id, int $status): bool
    {
        $sql = 'UPDATE slas SET is_active = :is_active, updated_at = NOW() WHERE id = :id AND deleted_at IS NULL';
        $statement = Database::run($sql, [
            'id' => $id,
            'is_active' => $status,
        ]);

        return $statement->rowCount() > 0;
    }

    public function nameExists(string $name, ?int $ignoreId = null): bool
    {
        $sql = 'SELECT 1 FROM slas WHERE name = :name AND deleted_at IS NULL';
        $params = ['name' => $name];

        if ($ignoreId !== null) {
            $sql .= ' AND id <> :ignore_id';
            $params['ignore_id'] = $ignoreId;
        }

        $statement = Database::run($sql, $params);
        return $statement->fetchColumn() !== false;
    }
}
