<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class TicketFieldRepository
{
    /**
     * @var array<int, array<int, array<string, mixed>>>
     */
    private array $catalogCache = [];

    public function fieldsByCategory(int $categoryId): array
    {
        if (isset($this->catalogCache[$categoryId])) {
            return $this->catalogCache[$categoryId];
        }

        $sql = <<<SQL
            SELECT
                id,
                category_id,
                name,
                label,
                type,
                options,
                is_required,
                sort_order
            FROM ticket_fields
            WHERE category_id = :category_id
              AND is_active = 1
            ORDER BY sort_order ASC, id ASC
        SQL;

        $statement = Database::run($sql, ['category_id' => $categoryId]);
        $rows = $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $this->catalogCache[$categoryId] = $this->hydrateFields($rows);

        return $this->catalogCache[$categoryId];
    }

    /**
     * @return array<int, array<int, array<string, mixed>>>
     */
    public function fieldsGroupedByCategory(): array
    {
        if ($this->catalogCache !== []) {
            return $this->catalogCache;
        }

        $sql = <<<SQL
            SELECT
                id,
                category_id,
                name,
                label,
                type,
                options,
                is_required,
                sort_order
            FROM ticket_fields
            WHERE is_active = 1
            ORDER BY category_id ASC, sort_order ASC, id ASC
        SQL;

        $statement = Database::run($sql);
        $rows = $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $grouped = [];

        foreach ($rows as $row) {
            $categoryId = (int) $row['category_id'];
            $grouped[$categoryId][] = $row;
        }

        foreach ($grouped as $categoryId => $items) {
            $this->catalogCache[$categoryId] = $this->hydrateFields($items);
        }

        return $this->catalogCache;
    }

    /**
     * @return array<string, string>
     */
    public function valueMapForTicket(int $ticketId): array
    {
        $sql = <<<SQL
            SELECT f.name, v.value
            FROM ticket_field_values v
            INNER JOIN ticket_fields f ON f.id = v.field_id
            WHERE v.ticket_id = :ticket_id
            ORDER BY f.sort_order ASC, f.id ASC
        SQL;

        $statement = Database::run($sql, ['ticket_id' => $ticketId]);
        $rows = $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $map = [];

        foreach ($rows as $row) {
            $map[$row['name']] = (string) $row['value'];
        }

        return $map;
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function displayFieldsForTicket(int $ticketId): array
    {
        $sql = <<<SQL
            SELECT
                f.id,
                f.label,
                f.name,
                f.type,
                v.value
            FROM ticket_field_values v
            INNER JOIN ticket_fields f ON f.id = v.field_id
            WHERE v.ticket_id = :ticket_id
            ORDER BY f.sort_order ASC, f.id ASC
        SQL;

        $statement = Database::run($sql, ['ticket_id' => $ticketId]);
        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function allWithCategory(): array
    {
        $sql = <<<SQL
            SELECT
                f.id,
                f.category_id,
                f.name,
                f.label,
                f.type,
                f.options,
                f.is_required,
                f.sort_order,
                c.name AS category_name
            FROM ticket_fields f
            INNER JOIN categories c ON c.id = f.category_id
            WHERE f.is_active = 1
            ORDER BY c.name ASC, f.sort_order ASC, f.id ASC
        SQL;

        $statement = Database::run($sql);
        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array{rows: array<int, array<string, mixed>>, total: int}
     */
    public function paginate(?string $search, ?string $status, int $limit, int $offset): array
    {
        $conditions = ['1=1'];
        $params = [];

        if ($search !== null && $search !== '') {
            $conditions[] = '(f.name LIKE :search OR f.label LIKE :search)';
            $params['search'] = '%' . $search . '%';
        }

        if ($status === 'active') {
            $conditions[] = 'f.is_active = 1';
        } elseif ($status === 'inactive') {
            $conditions[] = 'f.is_active = 0';
        }

        $where = 'WHERE ' . implode(' AND ', $conditions);

        $countSql = <<<SQL
            SELECT COUNT(*)
            FROM ticket_fields f
            INNER JOIN categories c ON c.id = f.category_id
            $where
        SQL;

        $countStmt = Database::prepare($countSql);
        foreach ($params as $key => $value) {
            $countStmt->bindValue(':' . $key, $value);
        }
        $countStmt->execute();
        $total = (int) $countStmt->fetchColumn();

        $sql = <<<SQL
            SELECT
                f.*,
                c.name AS category_name
            FROM ticket_fields f
            INNER JOIN categories c ON c.id = f.category_id
            $where
            ORDER BY c.name ASC, f.sort_order ASC, f.id ASC
            LIMIT :limit OFFSET :offset
        SQL;

        $statement = Database::connection()->prepare($sql);
        foreach ($params as $key => $value) {
            $statement->bindValue(':' . $key, $value);
        }
        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->bindValue(':offset', $offset, PDO::PARAM_INT);
        $statement->execute();

        return [
            'rows' => $statement->fetchAll(PDO::FETCH_ASSOC) ?: [],
            'total' => $total,
        ];
    }

    public function findById(int $id): ?array
    {
        $sql = <<<SQL
            SELECT
                f.*,
                c.name AS category_name
            FROM ticket_fields f
            INNER JOIN categories c ON c.id = f.category_id
            WHERE f.id = :id
            LIMIT 1
        SQL;

        $statement = Database::run($sql, ['id' => $id]);
        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? null : $row;
    }

    /**
     * @param array<string, mixed> $data
     */
    public function create(array $data): int
    {
        $sql = <<<SQL
            INSERT INTO ticket_fields (category_id, name, label, type, options, is_required, sort_order, is_active)
            VALUES (:category_id, :name, :label, :type, :options, :is_required, :sort_order, :is_active)
        SQL;

        $payload = array_merge([
            'options' => null,
            'sort_order' => 0,
            'is_active' => 1,
        ], $data);

        Database::run($sql, $payload);
        $this->catalogCache = [];

        return (int) Database::connection()->lastInsertId();
    }

    /**
     * @param array<string, mixed> $data
     */
    public function update(int $id, array $data): bool
    {
        $fields = [];
        foreach ($data as $column => $_value) {
            $fields[] = sprintf('%s = :%s', $column, $column);
        }
        $fields[] = 'updated_at = CURRENT_TIMESTAMP';

        $sql = sprintf(
            'UPDATE ticket_fields SET %s WHERE id = :id',
            implode(', ', $fields)
        );

        $data['id'] = $id;
        $statement = Database::run($sql, $data);
        $this->catalogCache = [];

        return $statement->rowCount() > 0;
    }

    public function delete(int $id): bool
    {
        $statement = Database::run('DELETE FROM ticket_fields WHERE id = :id', ['id' => $id]);
        $this->catalogCache = [];

        return $statement->rowCount() > 0;
    }

    public function isInUse(int $id): bool
    {
        $statement = Database::run(
            'SELECT 1 FROM ticket_field_values WHERE field_id = :id LIMIT 1',
            ['id' => $id]
        );

        return $statement->fetchColumn() !== false;
    }

    public function toggleStatus(int $id, int $status): bool
    {
        $statement = Database::run(
            'UPDATE ticket_fields SET is_active = :is_active, updated_at = NOW() WHERE id = :id',
            [
                'id' => $id,
                'is_active' => $status,
            ]
        );
        $this->catalogCache = [];

        return $statement->rowCount() > 0;
    }

    public function nameExists(int $categoryId, string $name, ?int $ignoreId = null): bool
    {
        $sql = <<<SQL
            SELECT 1
            FROM ticket_fields
            WHERE category_id = :category_id
              AND name = :name
        SQL;

        $params = [
            'category_id' => $categoryId,
            'name' => $name,
        ];

        if ($ignoreId !== null) {
            $sql .= ' AND id <> :ignore_id';
            $params['ignore_id'] = $ignoreId;
        }

        $sql .= ' LIMIT 1';

        $statement = Database::run($sql, $params);
        return $statement->fetchColumn() !== false;
    }

    public function deleteValuesForTicket(int $ticketId): void
    {
        Database::run(
            'DELETE FROM ticket_field_values WHERE ticket_id = :ticket_id',
            ['ticket_id' => $ticketId]
        );
    }

    /**
     * @param array<int, string> $values
     */
    public function saveValues(int $ticketId, array $values): void
    {
        foreach ($values as $fieldId => $value) {
            Database::run(
                <<<SQL
                    INSERT INTO ticket_field_values (ticket_id, field_id, value)
                    VALUES (:ticket_id, :field_id, :value)
                    ON DUPLICATE KEY UPDATE
                        value = VALUES(value),
                        updated_at = CURRENT_TIMESTAMP
                SQL,
                [
                    'ticket_id' => $ticketId,
                    'field_id' => $fieldId,
                    'value' => $value,
                ]
            );
        }
    }

    /**
     * @param array<int, array<string, mixed>> $rows
     * @return array<int, array<string, mixed>>
     */
    private function hydrateFields(array $rows): array
    {
        $fields = [];

        foreach ($rows as $row) {
            $options = $row['options'] ?? null;
            $decoded = [];

            if ($options !== null && $options !== '') {
                $parsed = json_decode((string) $options, true);
                if (is_array($parsed)) {
                    $decoded = array_values(array_map('strval', $parsed));
                }
            }

            $fields[] = [
                'id' => (int) $row['id'],
                'category_id' => (int) $row['category_id'],
                'name' => (string) $row['name'],
                'label' => (string) $row['label'],
                'type' => (string) $row['type'],
                'options' => $decoded,
                'is_required' => (int) $row['is_required'] === 1,
                'sort_order' => (int) ($row['sort_order'] ?? 0),
            ];
        }

        return $fields;
    }
}
