<?php
declare(strict_types=1);

namespace App\Repositories;

use App\Config\Database;
use PDO;

final class TicketRepository
{
    /**
     * @param array<string, mixed> $filters
     * @return array<int, array<string, mixed>>
     */
    public function listByFilters(array $filters = [], int $limit = 25, int $offset = 0): array
    {
        $sql = <<<SQL
            SELECT
                t.id,
                t.reference,
                t.client_type,
                t.main_category,
                t.site_name,
                t.condominium,
                t.subject,
                t.queue_id,
                q.name AS queue_name,
                t.requester_id,
                requester.name AS requester_name,
                requester.email AS requester_email,
                t.assignee_id,
                assignee.name AS assignee_name,
                assignee.email AS assignee_email,
                t.status_id,
                s.name AS status_name,
                s.slug AS status_slug,
                t.category_id,
                cat.name AS category_name,
                cat.slug AS category_slug,
                t.priority_id,
                p.name AS priority_name,
                t.created_at,
                t.updated_at,
                t.resolved_at,
                t.closed_at
            FROM tickets t
            INNER JOIN queues q ON q.id = t.queue_id
            INNER JOIN users requester ON requester.id = t.requester_id
            LEFT JOIN users assignee ON assignee.id = t.assignee_id
            INNER JOIN statuses s ON s.id = t.status_id
            INNER JOIN priorities p ON p.id = t.priority_id
            LEFT JOIN categories cat ON cat.id = t.category_id
        SQL;

        $conditions = [];
        $params = [];

        if (isset($filters['queue_id'])) {
            $conditions[] = 't.queue_id = :queue_id';
            $params['queue_id'] = (int) $filters['queue_id'];
        }

        if (isset($filters['requester_id'])) {
            $conditions[] = 't.requester_id = :requester_id';
            $params['requester_id'] = (int) $filters['requester_id'];
        }

        if (isset($filters['status_id'])) {
            $conditions[] = 't.status_id = :status_id';
            $params['status_id'] = (int) $filters['status_id'];
        }

        if (isset($filters['priority_id'])) {
            $conditions[] = 't.priority_id = :priority_id';
            $params['priority_id'] = (int) $filters['priority_id'];
        }

        if (isset($filters['category_id'])) {
            $conditions[] = 't.category_id = :category_id';
            $params['category_id'] = (int) $filters['category_id'];
        }

        if (!empty($filters['unassigned'])) {
            $conditions[] = 't.assignee_id IS NULL';
        } elseif (isset($filters['assignee_id'])) {
            $conditions[] = 't.assignee_id = :assignee_id';
            $params['assignee_id'] = (int) $filters['assignee_id'];
        }

        if (isset($filters['main_category'])) {
            $conditions[] = 't.main_category = :main_category';
            $params['main_category'] = $filters['main_category'];
        }

        if (isset($filters['from_date'])) {
            $conditions[] = 'DATE(t.created_at) >= :from_date';
            $params['from_date'] = $filters['from_date'];
        }

        if (isset($filters['to_date'])) {
            $conditions[] = 'DATE(t.created_at) <= :to_date';
            $params['to_date'] = $filters['to_date'];
        }

        if ($conditions !== []) {
            $sql .= ' WHERE ' . implode(' AND ', $conditions);
        }

        $sql .= ' ORDER BY t.created_at DESC LIMIT :limit OFFSET :offset';

        $statement = Database::connection()->prepare($sql);

        foreach ($params as $name => $value) {
            $type = is_int($value) ? PDO::PARAM_INT : PDO::PARAM_STR;
            $statement->bindValue(':' . $name, $value, $type);
        }

        $statement->bindValue(':limit', $limit, PDO::PARAM_INT);
        $statement->bindValue(':offset', $offset, PDO::PARAM_INT);
        $statement->execute();

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    public function findById(int $ticketId): ?array
    {
        $sql = <<<SQL
            SELECT
                t.id,
                t.reference,
                t.client_type,
                t.main_category,
                t.site_name,
                t.unit_name,
                t.condominium,
                t.customer_name,
                t.contact_name,
                t.contact_phone,
                t.contact_email,
                t.subject,
                t.queue_id,
                q.name AS queue_name,
                t.requester_id,
                requester.name AS requester_name,
                requester.email AS requester_email,
                t.assignee_id,
                assignee.name AS assignee_name,
                assignee.email AS assignee_email,
                t.status_id,
                s.name AS status_name,
                s.slug AS status_slug,
                t.priority_id,
                p.name AS priority_name,
                t.category_id,
                cat.name AS category_name,
                cat.slug AS category_slug,
                t.description,
                t.created_at,
                t.updated_at,
                t.resolved_at,
                t.closed_at
            FROM tickets t
            INNER JOIN queues q ON q.id = t.queue_id
            INNER JOIN users requester ON requester.id = t.requester_id
            LEFT JOIN users assignee ON assignee.id = t.assignee_id
            INNER JOIN statuses s ON s.id = t.status_id
            INNER JOIN priorities p ON p.id = t.priority_id
            LEFT JOIN categories cat ON cat.id = t.category_id
            WHERE t.id = :ticket_id
            LIMIT 1
        SQL;

        $statement = Database::run($sql, ['ticket_id' => $ticketId]);
        $ticket = $statement->fetch(PDO::FETCH_ASSOC);

        return $ticket === false ? null : $ticket;
    }

    /**
     * @param array<string, mixed> $data
     */
    public function create(array $data): int
    {
        $columns = array_keys($data);
        $placeholders = array_map(static fn ($column) => ':' . $column, $columns);
        $sql = sprintf(
            'INSERT INTO tickets (%s) VALUES (%s)',
            implode(', ', $columns),
            implode(', ', $placeholders)
        );

        Database::run($sql, $data);

        return (int) Database::connection()->lastInsertId();
    }

    /**
     * @param array<string, mixed> $data
     */
    public function update(int $ticketId, array $data): void
    {
        $set = [];
        foreach ($data as $column => $_) {
            $set[] = sprintf('%s = :%s', $column, $column);
        }

        $sql = sprintf('UPDATE tickets SET %s WHERE id = :id', implode(', ', $set));
        $data['id'] = $ticketId;

        Database::run($sql, $data);
    }

    public function delete(int $ticketId): void
    {
        Database::run('DELETE FROM tickets WHERE id = :id', ['id' => $ticketId]);
    }

    public function generateReference(): string
    {
        $sql = 'SELECT LPAD(COALESCE(MAX(id), 0) + 1, 4, \'0\') AS sequence FROM tickets';
        $statement = Database::run($sql);
        $row = $statement->fetch(PDO::FETCH_ASSOC);
        $sequence = $row['sequence'] ?? '0001';

        return 'HDK-' . $sequence;
    }

    public function insertHistory(int $ticketId, ?int $fromStatus, int $toStatus, ?int $userId, ?string $notes = null): void
    {
        $sql = <<<SQL
            INSERT INTO ticket_status_history (ticket_id, from_status_id, to_status_id, changed_by, notes)
            VALUES (:ticket_id, :from_status_id, :to_status_id, :changed_by, :notes)
        SQL;

        Database::run($sql, [
            'ticket_id' => $ticketId,
            'from_status_id' => $fromStatus,
            'to_status_id' => $toStatus,
            'changed_by' => $userId,
            'notes' => $notes,
        ]);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function statusHistory(int $ticketId): array
    {
        $sql = <<<SQL
            SELECT
                h.id,
                h.ticket_id,
                h.notes,
                h.changed_at,
                from_status.name AS from_status_name,
                from_status.slug AS from_status_slug,
                to_status.name AS to_status_name,
                to_status.slug AS to_status_slug,
                user_account.name AS changed_by_name
            FROM ticket_status_history h
            LEFT JOIN statuses from_status ON from_status.id = h.from_status_id
            LEFT JOIN statuses to_status ON to_status.id = h.to_status_id
            LEFT JOIN users user_account ON user_account.id = h.changed_by
            WHERE h.ticket_id = :ticket_id
            ORDER BY h.changed_at DESC
        SQL;

        $statement = Database::run($sql, ['ticket_id' => $ticketId]);

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    public function statusSlugById(int $statusId): ?string
    {
        $statement = Database::run(
            'SELECT slug FROM statuses WHERE id = :id LIMIT 1',
            ['id' => $statusId]
        );
        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? null : (string) $row['slug'];
    }

    public function statusIdBySlug(string $slug): ?int
    {
        $statement = Database::run(
            'SELECT id FROM statuses WHERE slug = :slug LIMIT 1',
            ['slug' => $slug]
        );
        $row = $statement->fetch(PDO::FETCH_ASSOC);

        return $row === false ? null : (int) $row['id'];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function ticketsToAutoClose(int $resolvedStatusId, string $limitTimestamp): array
    {
        $sql = <<<SQL
            SELECT id, status_id, resolved_at
            FROM tickets
            WHERE status_id = :status_id
              AND resolved_at IS NOT NULL
              AND resolved_at <= :limit_timestamp
        SQL;

        $statement = Database::connection()->prepare($sql);
        $statement->bindValue(':status_id', $resolvedStatusId, PDO::PARAM_INT);
        $statement->bindValue(':limit_timestamp', $limitTimestamp);
        $statement->execute();

        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function queueStatusSummary(int $queueId): array
    {
        $sql = <<<SQL
            SELECT s.slug, s.name, COUNT(*) AS total
            FROM tickets t
            INNER JOIN statuses s ON s.id = t.status_id
            WHERE t.queue_id = :queue_id
            GROUP BY s.slug, s.name
            ORDER BY s.id ASC
        SQL;

        $statement = Database::run($sql, ['queue_id' => $queueId]);
        return $statement->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
}
