<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\AdminAuditRepository;
use App\Repositories\CategoryRepository;
use App\Repositories\QueueRepository;
use RuntimeException;

final class CategoryService
{
    public function __construct(
        private readonly CategoryRepository $categories,
        private readonly QueueRepository $queues,
        private readonly AdminAuditRepository $audit
    ) {
    }

    /**
     * @return array{data: array<int, array<string, mixed>>, pagination: array<string, int>, filters: array<string, mixed>}
     */
    public function getPaginated(?string $search, ?string $status, int $page, int $perPage = 10): array
    {
        $search = $search !== null ? trim($search) : null;
        $supportsStatus = $this->categories->supportsStatus();
        $status = in_array($status, ['active', 'inactive'], true) ? $status : null;
        if (!$supportsStatus) {
            $status = null;
        }
        $page = max(1, $page);

        $offset = ($page - 1) * $perPage;
        $result = $this->categories->paginate($search, $status, $perPage, $offset);

        $pages = (int) max(1, ceil($result['total'] / $perPage));
        if ($page > $pages && $result['total'] > 0) {
            $page = $pages;
            $offset = ($page - 1) * $perPage;
            $result = $this->categories->paginate($search, $status, $perPage, $offset);
        }

        return [
            'data' => $result['rows'],
            'pagination' => [
                'page' => $page,
                'pages' => $pages,
                'per_page' => $perPage,
                'total' => $result['total'],
            ],
            'filters' => [
                'search' => $search,
                'status' => $supportsStatus ? ($status ?? 'all') : 'all',
            ],
        ];
    }

    public function findById(int $id): ?array
    {
        return $this->categories->findById($id);
    }

    public function isInUse(int $id): bool
    {
        return $this->categories->isInUse($id);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function parentOptions(?int $excludeId = null, ?int $includeId = null): array
    {
        return $this->categories->parentOptions($excludeId, $includeId);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function queueOptions(): array
    {
        return $this->queues->all();
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed>|null $actor
     */
    public function create(array $input, ?array $actor = null): int
    {
        $data = $this->validatePayload($input);
        $categoryId = $this->categories->create($data);
        $this->log($actor, 'create', $categoryId, [
            'name' => $data['name'],
            'slug' => $data['slug'],
        ]);

        return $categoryId;
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed>|null $actor
     */
    public function update(int $id, array $input, ?array $actor = null): void
    {
        $record = $this->findById($id);
        if ($record === null) {
            throw new RuntimeException('Categoria não encontrada.');
        }

        $data = $this->validatePayload($input, $id);
        if (!$this->categories->update($id, $data)) {
            throw new RuntimeException('Não foi possível atualizar a categoria informada.');
        }

        $this->log($actor, 'update', $id, [
            'name' => $data['name'],
            'slug' => $data['slug'],
        ]);
    }

    public function delete(int $id, ?array $actor = null): void
    {
        if ($this->findById($id) === null) {
            throw new RuntimeException('Categoria não encontrada.');
        }

        if ($this->categories->isInUse($id)) {
            throw new RuntimeException('Esta categoria possui vínculos e não pode ser excluída. Considere inativá-la.');
        }

        if (!$this->categories->delete($id)) {
            throw new RuntimeException('Categoria não encontrada ou já removida.');
        }

        $this->log($actor, 'delete', $id);
    }

    public function toggleStatus(int $id, bool $activate, ?array $actor = null): void
    {
        if ($this->findById($id) === null) {
            throw new RuntimeException('Categoria não encontrada.');
        }

        if (!$this->categories->supportsStatus()) {
            throw new RuntimeException('Alterar status requer a coluna is_active na tabela categories.');
        }

        if (!$this->categories->toggleStatus($id, $activate ? 1 : 0)) {
            throw new RuntimeException('Não foi possível alterar o status da categoria.');
        }

        $this->log($actor, 'toggle', $id, ['is_active' => $activate ? 1 : 0]);
    }

    /**
     * @param array<string, mixed> $input
     * @return array<string, mixed>
     */
    private function validatePayload(array $input, ?int $ignoreId = null): array
    {
        $name = trim((string) ($input['name'] ?? ''));
        if (mb_strlen($name) < 3) {
            throw new RuntimeException('O nome da categoria deve ter pelo menos 3 caracteres.');
        }

        $slugSource = (string) ($input['slug'] ?? $name);
        $slug = $this->slugify($slugSource);
        if ($slug === '') {
            throw new RuntimeException('Informe um slug válido (apenas letras, números, hífen e underline).');
        }

        if ($this->categories->nameExists($name, $ignoreId)) {
            throw new RuntimeException('Já existe uma categoria com esse nome.');
        }

        if ($this->categories->slugExists($slug, $ignoreId)) {
            throw new RuntimeException('Já existe uma categoria com esse slug.');
        }

        $parentRaw = $input['parent_id'] ?? null;
        if (is_string($parentRaw)) {
            $parentRaw = trim($parentRaw);
        }
        if ($parentRaw === 0 || $parentRaw === '0') {
            $parentRaw = null;
        }
        $parentId = $this->nullableInt($parentRaw);
        if ($parentId !== null) {
            if ($ignoreId !== null && $parentId === $ignoreId) {
                throw new RuntimeException('A categoria não pode ser pai de si mesma.');
            }

            $parent = $this->findById($parentId);
            if ($parent === null) {
                throw new RuntimeException('Categoria pai não encontrada.');
            }
        }

        $queueId = $this->nullableInt($input['queue_id'] ?? null);
        if ($queueId !== null && $this->queues->findById($queueId) === null) {
            throw new RuntimeException('Fila selecionada não existe.');
        }

        $sortOrder = isset($input['sort_order']) ? max(0, (int) $input['sort_order']) : 0;
        $description = trim((string) ($input['description'] ?? '')) ?: null;
        $isActive = isset($input['is_active']) ? 1 : 0;

        return [
            'name' => $name,
            'slug' => $slug,
            'description' => $description,
            'parent_id' => $parentId,
            'queue_id' => $queueId,
            'sort_order' => $sortOrder,
            'is_active' => $isActive,
        ];
    }

    private function slugify(string $value): string
    {
        $value = strtolower(trim($value));
        $value = preg_replace('/[^a-z0-9_-]+/i', '-', $value) ?? '';

        return trim($value, '-_');
    }

    private function nullableInt(mixed $value): ?int
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return (int) $value;
        }

        return null;
    }

    /**
     * @param array<string, mixed>|null $actor
     * @param array<string, mixed> $payload
     */
    private function log(?array $actor, string $action, ?int $entityId, array $payload = []): void
    {
        $userId = isset($actor['id']) ? (int) $actor['id'] : null;
        \app_log('INFO', 'Category action', [
            'user_id' => $userId,
            'action' => $action,
            'entity_id' => $entityId,
        ]);

        $this->audit->log($userId, 'category', $action, $entityId, $payload);
    }
}
