<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\AdminAuditRepository;
use App\Repositories\CustomerTypeRepository;
use RuntimeException;

final class CustomerTypeService
{
    public function __construct(
        private readonly CustomerTypeRepository $customerTypes,
        private readonly AdminAuditRepository $audit
    ) {
    }

    /**
     * @return array{data: array<int, array<string, mixed>>, pagination: array<string, int>, filters: array<string, mixed>}
     */
    public function getPaginated(?string $search, ?string $status, int $page, int $perPage = 10): array
    {
        $search = $search !== null ? trim($search) : null;
        $status = in_array($status, ['active', 'inactive'], true) ? $status : null;
        $page = max(1, $page);
        $offset = ($page - 1) * $perPage;

        $result = $this->customerTypes->paginate($search, $status, $perPage, $offset);
        $pages = (int) max(1, ceil($result['total'] / $perPage));
        if ($page > $pages && $result['total'] > 0) {
            $page = $pages;
            $offset = ($page - 1) * $perPage;
            $result = $this->customerTypes->paginate($search, $status, $perPage, $offset);
        }

        return [
            'data' => $result['rows'],
            'pagination' => [
                'page' => $page,
                'pages' => $pages,
                'per_page' => $perPage,
                'total' => $result['total'],
            ],
            'filters' => [
                'search' => $search ?? '',
                'status' => $status ?? 'all',
            ],
        ];
    }

    public function findById(int $id): ?array
    {
        return $this->customerTypes->findById($id);
    }

    public function isInUse(int $id): bool
    {
        return $this->customerTypes->isInUse($id);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function activeOptions(): array
    {
        return $this->customerTypes->active();
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed>|null $actor
     */
    public function create(array $input, ?array $actor = null): int
    {
        $data = $this->validatePayload($input);
        $id = $this->customerTypes->create($data);
        $this->log($actor, 'create', $id, [
            'slug' => $data['slug'],
            'name' => $data['name'],
        ]);

        return $id;
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed>|null $actor
     */
    public function update(int $id, array $input, ?array $actor = null): void
    {
        if ($this->findById($id) === null) {
            throw new RuntimeException('Tipo de cliente não encontrado.');
        }

        $data = $this->validatePayload($input, $id);
        if (!$this->customerTypes->update($id, $data)) {
            throw new RuntimeException('Não foi possível atualizar o registro informado.');
        }

        $this->log($actor, 'update', $id, [
            'slug' => $data['slug'],
            'name' => $data['name'],
        ]);
    }

    public function delete(int $id, ?array $actor = null): void
    {
        if ($this->findById($id) === null) {
            throw new RuntimeException('Tipo de cliente não encontrado.');
        }

        if ($this->customerTypes->isInUse($id)) {
            throw new RuntimeException('Este tipo já está em uso; inative ao invés de excluir.');
        }

        if (!$this->customerTypes->delete($id)) {
            throw new RuntimeException('Registro não encontrado ou já removido.');
        }

        $this->log($actor, 'delete', $id);
    }

    public function toggleStatus(int $id, bool $activate, ?array $actor = null): void
    {
        if ($this->findById($id) === null) {
            throw new RuntimeException('Tipo de cliente não encontrado.');
        }

        if (!$this->customerTypes->toggleStatus($id, $activate ? 1 : 0)) {
            throw new RuntimeException('Não foi possível alterar o status informado.');
        }

        $this->log($actor, 'toggle', $id, ['is_active' => $activate ? 1 : 0]);
    }

    /**
     * @param array<string, mixed> $input
     * @return array<string, mixed>
     */
    private function validatePayload(array $input, ?int $ignoreId = null): array
    {
        $name = trim((string) ($input['name'] ?? ''));
        if ($name === '' || mb_strlen($name) < 3) {
            throw new RuntimeException('O nome deve ter ao menos 3 caracteres.');
        }

        $slug = $this->slugify((string) ($input['slug'] ?? $name));
        if ($slug === '') {
            throw new RuntimeException('Informe um identificador válido.');
        }

        if ($this->customerTypes->nameExists($name, $ignoreId)) {
            throw new RuntimeException('Já existe um tipo de cliente com esse nome.');
        }

        if ($this->customerTypes->slugExists($slug, $ignoreId)) {
            throw new RuntimeException('Já existe um tipo de cliente com esse identificador.');
        }

        $sortOrder = (int) ($input['sort_order'] ?? 0);
        $description = trim((string) ($input['description'] ?? ''));

        return [
            'name' => $name,
            'slug' => $slug,
            'description' => $description !== '' ? $description : null,
            'sort_order' => $sortOrder,
            'is_active' => isset($input['is_active']) ? 1 : 0,
        ];
    }

    private function slugify(string $value): string
    {
        $value = strtolower(trim($value));
        $value = preg_replace('/[^a-z0-9\-_]+/', '-', $value) ?? '';

        return trim($value, '-');
    }

    /**
     * @param array<string, mixed>|null $actor
     * @param array<string, mixed> $changes
     */
    private function log(?array $actor, string $action, ?int $entityId = null, array $changes = []): void
    {
        $this->audit->log('customer_type', $action, $actor['id'] ?? null, $entityId, $changes);
    }
}
