<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\AdminAuditRepository;
use App\Repositories\QueueRepository;
use RuntimeException;

final class QueueService
{
    public function __construct(
        private readonly QueueRepository $queues,
        private readonly AdminAuditRepository $audit
    ) {
    }

    /**
     * @return array{data: array<int, array<string, mixed>>, pagination: array<string, int>, filters: array<string, mixed>}
     */
    public function getPaginated(?string $search, ?string $status, int $page, int $perPage = 10): array
    {
        $search = $search !== null ? trim($search) : null;
        $status = in_array($status, ['active', 'inactive'], true) ? $status : null;
        $page = max(1, $page);

        $offset = ($page - 1) * $perPage;
        $result = $this->queues->paginate($search, $status, $perPage, $offset);

        $pages = (int) max(1, ceil($result['total'] / $perPage));
        if ($page > $pages && $result['total'] > 0) {
            $page = $pages;
            $offset = ($page - 1) * $perPage;
            $result = $this->queues->paginate($search, $status, $perPage, $offset);
        }

        return [
            'data' => $result['rows'],
            'pagination' => [
                'page' => $page,
                'pages' => $pages,
                'per_page' => $perPage,
                'total' => $result['total'],
            ],
            'filters' => [
                'search' => $search,
                'status' => $status ?? 'all',
            ],
        ];
    }

    public function findById(int $id): ?array
    {
        return $this->queues->findById($id);
    }

    public function isInUse(int $id): bool
    {
        return $this->queues->isInUse($id);
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed>|null $actor
     */
    public function create(array $input, ?array $actor = null): int
    {
        $data = $this->validatePayload($input);
        $queueId = $this->queues->create($data);
        $this->log($actor, 'create', $queueId, [
            'name' => $data['name'],
            'slug' => $data['slug'],
        ]);

        return $queueId;
    }

    /**
     * @param array<string, mixed> $input
     * @param array<string, mixed>|null $actor
     */
    public function update(int $id, array $input, ?array $actor = null): void
    {
        $record = $this->findById($id);
        if ($record === null) {
            throw new RuntimeException('Fila não encontrada.');
        }

        $data = $this->validatePayload($input, $id);
        if (!$this->queues->update($id, $data)) {
            throw new RuntimeException('Não foi possível atualizar a fila informada.');
        }

        $this->log($actor, 'update', $id, [
            'name' => $data['name'],
            'slug' => $data['slug'],
        ]);
    }

    public function delete(int $id, ?array $actor = null): void
    {
        if ($this->isInUse($id)) {
            throw new RuntimeException('Registro em uso. Não pode excluir. Inative.');
        }

        if (!$this->queues->delete($id)) {
            throw new RuntimeException('Fila não encontrada ou já removida.');
        }

        $this->log($actor, 'delete', $id);
    }

    public function toggleStatus(int $id, bool $activate, ?array $actor = null): void
    {
        if (!$this->queues->toggleStatus($id, $activate ? 1 : 0)) {
            throw new RuntimeException('Não foi possível alterar o status da fila.');
        }

        $this->log($actor, 'toggle', $id, ['is_active' => $activate ? 1 : 0]);
    }

    public function nameExists(string $name, ?int $ignoreId = null): bool
    {
        return $this->queues->nameExists($name, $ignoreId);
    }

    public function slugExists(string $slug, ?int $ignoreId = null): bool
    {
        return $this->queues->slugExists($slug, $ignoreId);
    }

    /**
     * @param array<string, mixed> $input
     * @return array<string, mixed>
     */
    private function validatePayload(array $input, ?int $ignoreId = null): array
    {
        $name = trim((string) ($input['name'] ?? ''));
        if (mb_strlen($name) < 3) {
            throw new RuntimeException('O nome da fila deve ter pelo menos 3 caracteres.');
        }

        $slugSource = (string) ($input['slug'] ?? $name);
        $slug = $this->slugify($slugSource);
        if ($slug === '') {
            throw new RuntimeException('Informe um slug válido (apenas letras, números, hífen e underline).');
        }

        if ($this->nameExists($name, $ignoreId)) {
            throw new RuntimeException('Já existe uma fila com esse nome.');
        }

        if ($this->slugExists($slug, $ignoreId)) {
            throw new RuntimeException('Já existe uma fila com esse slug.');
        }

        $response = max(0, (int) ($input['sla_response_minutes'] ?? 0));
        $resolution = max(0, (int) ($input['sla_resolution_minutes'] ?? 0));

        return [
            'name' => $name,
            'slug' => $slug,
            'description' => trim((string) ($input['description'] ?? '')) ?: null,
            'sla_response_minutes' => $response,
            'sla_resolution_minutes' => $resolution,
            'is_active' => isset($input['is_active']) ? 1 : 0,
        ];
    }

    private function slugify(string $value): string
    {
        $value = strtolower(trim($value));
        $value = preg_replace('/[^a-z0-9_-]+/i', '-', $value) ?? '';

        return trim($value, '-_');
    }

    /**
     * @param array<string, mixed>|null $actor
     * @param array<string, mixed> $payload
     */
    private function log(?array $actor, string $action, ?int $entityId, array $payload = []): void
    {
        $userId = isset($actor['id']) ? (int) $actor['id'] : null;
        \app_log('INFO', 'Queue action', [
            'user_id' => $userId,
            'action' => $action,
            'entity_id' => $entityId,
        ]);

        $this->audit->log($userId, 'queue', $action, $entityId, $payload);
    }
}
