<?php
declare(strict_types=1);

namespace App\Services;

use App\Repositories\PriorityRepository;
use App\Repositories\SlaPolicyRepository;
use App\Repositories\SlaRepository;
use RuntimeException;

final class SlaService
{
    /**
     * Statuses that pause the SLA clock (awaiting user or third party).
     *
     * @var string[]
     */
    private array $pauseStatuses = ['aguardando-cliente', 'pausado-interno'];

    /**
     * Statuses that we consider a first response when transitioning into.
     *
     * @var string[]
     */
    private array $responseTriggerStatuses = [
        'em-triagem',
        'em-atendimento',
        'aguardando-cliente',
        'pausado-interno',
        'resolvido',
        'fechado',
    ];

    public function __construct(
        private readonly SlaPolicyRepository $policies,
        private readonly PriorityRepository $priorities,
        private readonly SlaRepository $slas
    ) {
    }

    /**
     * @param array<int, array<string, mixed>> $history
     * @return array<string, mixed>
     */
    public function metrics(array $ticket, array $history): array
    {
        $policy = $this->policyForTicket($ticket);
        $historyAsc = $this->sortHistory($history);

        return [
            'policy' => $policy,
            'response' => $this->calculateResponse($ticket, $historyAsc, $policy['response_minutes']),
            'resolution' => $this->calculateResolution($ticket, $historyAsc, $policy['resolution_minutes']),
        ];
    }

    /**
     * @return array<string, mixed>
     */
    private function policyForTicket(array $ticket): array
    {
        $categoryId = isset($ticket['category_id']) ? (int) $ticket['category_id'] : null;
        if ($categoryId === 0) {
            $categoryId = null;
        }

        $priorityId = (int) ($ticket['priority_id'] ?? 0);
        $clientType = (string) ($ticket['client_type'] ?? 'interno');

        $policy = $this->policies->findPolicy($categoryId, $priorityId, $clientType);
        if ($policy !== null) {
            return [
                'response_minutes' => (int) $policy['response_minutes'],
                'resolution_minutes' => (int) $policy['resolution_minutes'],
                'source' => 'policy',
            ];
        }

        $priority = $priorityId > 0 ? $this->priorities->findById($priorityId) : null;
        if ($priority !== null) {
            return [
                'response_minutes' => (int) ($priority['response_target_minutes'] ?? 0),
                'resolution_minutes' => (int) ($priority['resolution_target_minutes'] ?? 0),
                'source' => 'priority',
            ];
        }

        return [
            'response_minutes' => 0,
            'resolution_minutes' => 0,
            'source' => 'default',
        ];
    }

    /**
     * @param array<int, array<string, mixed>> $history
     */
    private function calculateResponse(array $ticket, array $history, int $targetMinutes): array
    {
        $createdAt = $this->timestamp($ticket['created_at'] ?? null);
        if ($createdAt === null) {
            return $this->emptyMetric($targetMinutes);
        }

        $firstResponseAt = $this->detectFirstResponse($ticket, $history, $createdAt);
        $endReference = $firstResponseAt ?? time();
        $elapsedMinutes = $this->minutesBetween($createdAt, $endReference);

        $status = 'pending';
        if ($firstResponseAt !== null) {
            $status = ($targetMinutes > 0 && $elapsedMinutes > $targetMinutes) ? 'late' : 'on_track';
        } elseif ($targetMinutes > 0 && $elapsedMinutes > $targetMinutes) {
            $status = 'late';
        }

        return [
            'target' => $targetMinutes,
            'elapsed' => $elapsedMinutes,
            'remaining' => $targetMinutes - $elapsedMinutes,
            'status' => $status,
            'achieved_at' => $firstResponseAt ? date('Y-m-d H:i:s', $firstResponseAt) : null,
        ];
    }

    /**
     * @param array<int, array<string, mixed>> $history
     */
    private function calculateResolution(array $ticket, array $history, int $targetMinutes): array
    {
        $createdAt = $this->timestamp($ticket['created_at'] ?? null);
        if ($createdAt === null) {
            return $this->emptyMetric($targetMinutes);
        }

        $endTime = $this->resolutionEndTimestamp($ticket);
        $segments = $this->buildSegments($ticket, $history, $createdAt);
        $elapsedMinutes = $this->workingMinutes($segments, $endTime);
        $isCompleted = !empty($ticket['resolved_at']) || !empty($ticket['closed_at']);

        $status = 'pending';
        if ($targetMinutes <= 0) {
            $status = 'not_configured';
        } elseif ($elapsedMinutes > $targetMinutes) {
            $status = 'late';
        } elseif ($isCompleted) {
            $status = 'on_track';
        }

        return [
            'target' => $targetMinutes,
            'elapsed' => $elapsedMinutes,
            'remaining' => $targetMinutes - $elapsedMinutes,
            'status' => $status,
            'completed_at' => $isCompleted ? date('Y-m-d H:i:s', $endTime) : null,
        ];
    }

    /**
     * @param array<int, array<string, mixed>> $history
     */
    private function detectFirstResponse(array $ticket, array $history, int $createdAt): ?int
    {
        $initialSlug = $history !== []
            ? ($history[0]['to_status_slug'] ?? null)
            : ($ticket['status_slug'] ?? null);

        if ($initialSlug !== null && $initialSlug !== 'novo') {
            return $createdAt;
        }

        foreach ($history as $event) {
            $from = $event['from_status_slug'] ?? null;
            $to = $event['to_status_slug'] ?? null;
            $timestamp = $this->timestamp($event['changed_at'] ?? null);
            if ($timestamp === null) {
                continue;
            }

            if ($from === 'novo' && $to !== 'novo') {
                return $timestamp;
            }

            if ($from === null && $to !== 'novo' && in_array($to, $this->responseTriggerStatuses, true)) {
                return $timestamp;
            }
        }

        return null;
    }

    /**
     * @param array<int, array<string, mixed>> $history
     * @return array<int, array<string, mixed>>
     */
    private function buildSegments(array $ticket, array $history, int $createdAt): array
    {
        $segments = [];
        $historyCount = count($history);
        $currentStatus = $historyCount > 0
            ? ($history[0]['to_status_slug'] ?? 'novo')
            : ($ticket['status_slug'] ?? 'novo');
        $segmentStart = $createdAt;

        if ($historyCount === 0) {
            $segments[] = [
                'status' => $currentStatus,
                'start' => $segmentStart,
                'end' => null,
            ];

            return $segments;
        }

        foreach ($history as $index => $event) {
            $eventTime = $this->timestamp($event['changed_at'] ?? null);
            if ($index === 0) {
                $currentStatus = $event['to_status_slug'] ?? $currentStatus;
                $segmentStart = $eventTime ?? $segmentStart;
                continue;
            }

            if ($eventTime === null) {
                continue;
            }

            $segments[] = [
                'status' => $currentStatus,
                'start' => $segmentStart,
                'end' => $eventTime,
            ];

            $currentStatus = $event['to_status_slug'] ?? $currentStatus;
            $segmentStart = $eventTime;
        }

        $segments[] = [
            'status' => $currentStatus,
            'start' => $segmentStart,
            'end' => null,
        ];

        return $segments;
    }

    /**
     * @param array<int, array<string, mixed>> $segments
     */
    private function workingMinutes(array $segments, int $endTime): int
    {
        $minutes = 0.0;

        foreach ($segments as $segment) {
            $start = (int) ($segment['start'] ?? 0);
            $end = $segment['end'] === null ? $endTime : (int) $segment['end'];
            if ($end > $endTime) {
                $end = $endTime;
            }

            if ($end <= $start) {
                continue;
            }

            $status = (string) ($segment['status'] ?? '');
            if (in_array($status, $this->pauseStatuses, true)) {
                continue;
            }

            $minutes += ($end - $start) / 60;
        }

        return (int) ceil(max(0, $minutes));
    }

    /**
     * @param array<int, array<string, mixed>> $history
     * @return array<int, array<string, mixed>>
     */
    private function sortHistory(array $history): array
    {
        usort(
            $history,
            static function (array $a, array $b): int {
                $first = $a['changed_at'] ?? '';
                $second = $b['changed_at'] ?? '';
                return strcmp((string) $first, (string) $second);
            }
        );

        return $history;
    }

    private function minutesBetween(int $start, int $end): int
    {
        if ($end <= $start) {
            return 0;
        }

        return (int) ceil(($end - $start) / 60);
    }

    private function timestamp(?string $value): ?int
    {
        if ($value === null || $value === '') {
            return null;
        }

        $time = strtotime($value);

        return $time === false ? null : $time;
    }

    private function resolutionEndTimestamp(array $ticket): int
    {
        $closed = $this->timestamp($ticket['closed_at'] ?? null);
        if ($closed !== null) {
            return $closed;
        }

        $resolved = $this->timestamp($ticket['resolved_at'] ?? null);
        if ($resolved !== null) {
            return $resolved;
        }

        return time();
    }

    /**
     * @return array<string, mixed>
     */
    private function emptyMetric(int $target): array
    {
        return [
            'target' => $target,
            'elapsed' => 0,
            'remaining' => $target,
            'status' => $target > 0 ? 'pending' : 'not_configured',
            'achieved_at' => null,
            'completed_at' => null,
        ];
    }

    /**
     * Retorna SLAs paginados aplicando busca e filtro de status.
     *
     * @return array{data: array<int, array<string, mixed>>, pagination: array<string, int>, filters: array<string, ?string>}
     */
    public function getPaginated(?string $search, ?string $status, int $page, int $perPage = 10): array
    {
        $search = $search !== null ? trim($search) : null;
        $status = in_array($status, ['active', 'inactive'], true) ? $status : null;
        $page = max(1, $page);

        $offset = ($page - 1) * $perPage;
        $result = $this->slas->paginate($search, $status, $perPage, $offset);

        $pages = (int) max(1, ceil($result['total'] / $perPage));
        if ($page > $pages && $result['total'] > 0) {
            $page = $pages;
            $offset = ($page - 1) * $perPage;
            $result = $this->slas->paginate($search, $status, $perPage, $offset);
        }

        return [
            'data' => $result['rows'],
            'pagination' => [
                'page' => $page,
                'pages' => $pages,
                'per_page' => $perPage,
                'total' => $result['total'],
            ],
            'filters' => [
                'search' => $search,
                'status' => $status ?? 'all',
            ],
        ];
    }

    public function find(int $id): ?array
    {
        return $this->slas->findById($id);
    }

    public function isInUse(int $id): bool
    {
        return $this->slas->isInUse($id);
    }

    /**
     * @param array<string, mixed> $input
     */
    public function create(array $input): int
    {
        $data = $this->validatePayload($input);
        return $this->slas->create($data);
    }

    /**
     * @param array<string, mixed> $input
     */
    public function update(int $id, array $input): void
    {
        $record = $this->find($id);
        if ($record === null) {
            throw new RuntimeException('SLA não encontrado.');
        }

        $data = $this->validatePayload($input, $id);
        if (!$this->slas->update($id, $data)) {
            throw new RuntimeException('Não foi possível atualizar o SLA informado.');
        }
    }

    public function delete(int $id): void
    {
        if ($this->isInUse($id)) {
            throw new RuntimeException('Registro em uso. Não pode excluir. Inative.');
        }

        if (!$this->slas->softDelete($id)) {
            throw new RuntimeException('SLA não encontrado ou já removido.');
        }
    }

    public function toggleStatus(int $id, bool $activate): void
    {
        if (!$this->slas->toggleStatus($id, $activate ? 1 : 0)) {
            throw new RuntimeException('Não foi possível alterar o status do SLA.');
        }
    }

    public function nameExists(string $name, ?int $ignoreId = null): bool
    {
        return $this->slas->nameExists($name, $ignoreId);
    }

    /**
     * @param array<string, mixed> $input
     * @return array<string, mixed>
     */
    private function validatePayload(array $input, ?int $ignoreId = null): array
    {
        $name = trim((string) ($input['name'] ?? ''));
        if (mb_strlen($name) < 3) {
            throw new RuntimeException('O nome do SLA deve ter pelo menos 3 caracteres.');
        }

        if ($this->nameExists($name, $ignoreId)) {
            throw new RuntimeException('Já existe um SLA com este nome.');
        }

        $firstResponse = (int) ($input['first_response_minutes'] ?? 0);
        $resolution = (int) ($input['resolution_minutes'] ?? 0);
        if ($firstResponse <= 0 || $resolution <= 0) {
            throw new RuntimeException('Defina tempos válidos para resposta e resolução.');
        }

        $businessHours = isset($input['business_hours_only']) ? 1 : 0;
        $escalationEnabled = isset($input['escalation_enabled']) ? 1 : 0;
        $escalationMinutes = null;
        if ($escalationEnabled === 1) {
            $escalationMinutes = (int) ($input['escalation_minutes'] ?? 0);
            if ($escalationMinutes <= 0) {
                throw new RuntimeException('Informe o tempo de escalonamento em minutos.');
            }
        }

        $priorityId = $this->normalizeInt($input['priority_id'] ?? null);
        if ($priorityId !== null && $this->priorities->findById($priorityId) === null) {
            throw new RuntimeException('Prioridade selecionada não existe.');
        }

        return [
            'name' => $name,
            'description' => trim((string) ($input['description'] ?? '')) ?: null,
            'first_response_minutes' => $firstResponse,
            'resolution_minutes' => $resolution,
            'business_hours_only' => $businessHours,
            'escalation_enabled' => $escalationEnabled,
            'escalation_minutes' => $escalationMinutes,
            'priority_id' => $priorityId,
            'is_active' => isset($input['is_active']) ? 1 : 0,
        ];
    }

    private function normalizeInt(mixed $value): ?int
    {
        if ($value === null || $value === '' || (int) $value === 0) {
            return null;
        }

        return (int) $value;
    }
}
