<?php
declare(strict_types=1);

namespace App\Services;

use App\Config\AppConfig;
use JsonException;
use RuntimeException;

final class SsoTokenService
{
    private string $storageFile;

    public function __construct()
    {
        $cacheDirectory = BASE_PATH . '/storage/cache';
        if (!is_dir($cacheDirectory)) {
            mkdir($cacheDirectory, 0775, true);
        }

        $this->storageFile = $cacheDirectory . '/sso_tokens.json';
        if (!file_exists($this->storageFile)) {
            file_put_contents($this->storageFile, json_encode([]), LOCK_EX);
        }
    }

    /**
     * @return array{email:string,expires:int,nonce:string}
     */
    public function validateToken(string $token): array
    {
        if ($token === '') {
            throw new RuntimeException('Token ausente.');
        }

        $parts = explode('.', $token, 2);
        if (count($parts) !== 2) {
            throw new RuntimeException('Token mal formatado.');
        }

        [$encodedPayload, $providedSignature] = $parts;
        $expectedSignature = $this->sign($encodedPayload);

        if (!hash_equals($expectedSignature, $providedSignature)) {
            throw new RuntimeException('Assinatura invalida.');
        }

        $payloadJson = $this->base64UrlDecode($encodedPayload);
        try {
            $payload = json_decode($payloadJson, true, 512, JSON_THROW_ON_ERROR);
        } catch (JsonException) {
            throw new RuntimeException('Payload invalido.');
        }

        if (!is_array($payload)) {
            throw new RuntimeException('Payload invalido.');
        }

        foreach (['email', 'expires', 'nonce'] as $required) {
            if (!array_key_exists($required, $payload)) {
                throw new RuntimeException('Token incompleto.');
            }
        }

        $expiresAt = (int) $payload['expires'];
        $now = time();
        $drift = AppConfig::ssoAllowedDrift();

        if ($expiresAt < ($now - $drift)) {
            throw new RuntimeException('Token expirado.');
        }

        if (($expiresAt - $now) > AppConfig::ssoTokenTtl()) {
            throw new RuntimeException('Janela do token invalida.');
        }

        $email = strtolower((string) $payload['email']);
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new RuntimeException('Email do token invalido.');
        }

        $nonce = (string) $payload['nonce'];
        if ($nonce === '') {
            throw new RuntimeException('Nonce do token invalido.');
        }

        $this->markNonceUsage($nonce, $expiresAt);

        return [
            'email' => $email,
            'expires' => $expiresAt,
            'nonce' => $nonce,
        ];
    }

    private function sign(string $payload): string
    {
        return $this->base64UrlEncode(hash_hmac('sha256', $payload, AppConfig::ssoSecret(), true));
    }

    private function base64UrlEncode(string $data): string
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    private function base64UrlDecode(string $data): string
    {
        $remainder = strlen($data) % 4;
        if ($remainder > 0) {
            $data .= str_repeat('=', 4 - $remainder);
        }

        $decoded = base64_decode(strtr($data, '-_', '+/'), true);
        if ($decoded === false) {
            throw new RuntimeException('Falha ao decodificar token.');
        }

        return $decoded;
    }

    private function markNonceUsage(string $nonce, int $expiresAt): void
    {
        $handle = fopen($this->storageFile, 'c+');
        if ($handle === false) {
            throw new RuntimeException('Nao foi possivel abrir armazenamento de tokens.');
        }

        try {
            if (!flock($handle, LOCK_EX)) {
                throw new RuntimeException('Nao foi possivel travar armazenamento de tokens.');
            }

            $contents = stream_get_contents($handle);
            $data = $contents ? json_decode($contents, true) : [];
            if (!is_array($data)) {
                $data = [];
            }

            $now = time();
            foreach ($data as $hash => $expiration) {
                if ((int) $expiration < $now) {
                    unset($data[$hash]);
                }
            }

            $nonceHash = hash('sha256', $nonce);
            if (isset($data[$nonceHash])) {
                throw new RuntimeException('Token ja utilizado.');
            }

            $data[$nonceHash] = $expiresAt;

            ftruncate($handle, 0);
            rewind($handle);
            fwrite($handle, json_encode($data, JSON_PRETTY_PRINT));
        } finally {
            flock($handle, LOCK_UN);
            fclose($handle);
        }
    }
}
